//
// chat.cpp
//

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/signal.h>
#include <sys/socket.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <sys/errno.h>
#include <netinet/in.h>
#include <stdio.h>

#define QLEN 10                                         // maximum connection queue length
#define BUFSIZE 4096                            // maximum amount of incoming data we can handle

extern int errno;
extern char *sys_errlist[];

int passivesock(char *, char *, int);
int errexit(char *, char *, char *);

/**
 * main()
 */

int main
(
        int argc,
        char *argv[]
)
{
        char *service = "6001";                 // service name or port number
        struct sockaddr_in fsin;                // the address of the client
        int msock;                                              // master server socket
        fd_set rfds;                                    // read file descriptor set
        fd_set afds;                                    // active file descriptor set
        int alen;                                               // length of client's address
        int fd, nfds;                                   // from-address length

        switch (argc)
        {
        case 1:
                printf("using port %s\n", service);
                break;
        case 2:
                service = argv[1];
                break;
        default:
                errexit("usage: chat [port]\n", "", "");
        }

        // Create the master server socket
        msock = passivesock(service, "tcp", QLEN);
        printf("Master socket = %d\n", msock);

        nfds = getdtablesize();
        printf("Up to %d connections\n", nfds);

        FD_ZERO(&afds);
        FD_SET(msock, &afds);

        for (;;)
        {
                bcopy((char *)&afds, (char *)&rfds, sizeof(rfds));

                // block until a new connection request is made, or there is new data
        printf("Waiting on select...\n");
                if (select(nfds, &rfds, (fd_set *)0, (fd_set *)0, (struct timeval *)0) < 0)
                        errexit("select: %s\n", sys_errlist[errno], "");

                if (FD_ISSET(msock, &rfds))
                {
                        // a new connection request; create a new socket for the connection
                        int ssock;
                        alen = sizeof(fsin);
                        ssock = accept(msock, (struct sockaddr *)&fsin, &alen);
                        if (ssock < 0)
                                errexit("accept: %s\n", sys_errlist[errno], "");
            printf("Accept on %d.\n", ssock);
                        FD_SET(ssock, &afds);
                }

                // Check all sockets for incoming data
                for (fd = 0; fd < nfds; ++fd)
                {
                        if ((fd != msock) && FD_ISSET(fd, &rfds))
                        {
                                // Read each input stream
                                char buf[BUFSIZE];
                                int cc = read(fd, buf, sizeof buf);
                                if (cc < 0)
                                {
                                        // client exited - clear this socket
                                        (void)close(fd);
                                        FD_CLR(fd, &afds);

                                        continue;
                                }
                                else if (cc == 0)
                                {
                                        // EOF - close connection with this client
                                        printf("Closing connection to %d\n", fd);
                                        (void)close(fd);
                                        FD_CLR(fd, &afds);
                                }
                                else
                                {
                                        // data received on fd

                                        // broadcast the data to all clients except current one
                                        for (int bfd = 0; bfd < nfds; ++bfd)
                                        {
                                                if ((bfd != msock) && FD_ISSET(bfd, &afds))
                                                {
                                                        if (write(bfd, buf, cc) < 0)
                                                                errexit("client write: %s\n", sys_errlist[errno], "");
                                                }
                                        }
                                }
                        }
                }
        }
}


#include <netdb.h>

u_short htons(), ntohs();

u_short portbase = 0;                           // port base, for non-root servers

/**
 * Create a server socket to listen to the port for session requests.
 */

int passivesock
(
        char *service,                                  // service or port (in our case "6001")
        char *protocol,                                 // name of protocol to use (in our case "tcp")
        int qlen                                                // maximum length of the server request queue
)
{
        struct servent *pse;                    // pointer to service information entry
        struct protoent *ppe;                   // pointer to protocol information entry
        struct sockaddr_in sin;                 // an Internet endpoint address
        int s, type;                                    // socket description and socket type

        bzero((char *)&sin, sizeof(sin));
        sin.sin_family = AF_INET;
        sin.sin_addr.s_addr = INADDR_ANY;

        // Map service name to port number
        if (pse = getservbyname(service, protocol))
                sin.sin_port = htons(ntohs((u_short)pse->s_port) + portbase);
        else if ((sin.sin_port = htons((u_short)atoi(service))) == 0)
                errexit("can't get \"%s\" service entry\n", service, "");

        // Map protocol name to protocol number
        if ((ppe = getprotobyname(protocol)) == 0)
                errexit("can't get \"%s\" protocol entry\n", protocol, "");

        // Use protocol to choose a socket type
        if (strcmp(protocol, "udp") == 0)
                type = SOCK_DGRAM;
        else
                type = SOCK_STREAM;

        // Allocate a socket
        s = socket(PF_INET, type, ppe->p_proto);
        if (s < 0)
                errexit("can't create socket: %s\n", sys_errlist[errno], "");

        // Bind the socket
        if (bind(s, (struct sockaddr *)&sin, sizeof(sin)) < 0)
                errexit("can't bind to %s port: %s\n", service, sys_errlist[errno]);

        // Put socket in passive mode
        if (type == SOCK_STREAM && listen(s, qlen) < 0)
                errexit("can't listen on %s port: %s\n", service, sys_errlist[errno]);
        
        return s;
}


int errexit
(
        char *format,
        char *a, char *b
)
{
        fprintf(stderr, format, a, b);
        printf(format, a, b);
        exit(1);
}

