#!/usr/local/bin/perl
######################################################################
#
#   Copyright 1999-2000 Phone.com, Inc.  All rights reserved.
#
#   Subject to the terms and conditions of the SDK License Agreement, 
#   Phone.com, Inc. hereby grants you the right to use the UP.SDK 
#   software and its related documentation.
#
#   The Phone.com name and logo and the family of terms carrying the "UP."
#   prefix are trademarks of Phone.com, Inc. All other brands and product 
#   names may be trademarks of the respective companies with which they 
#   are associated.
#
######################################################################

######################################################################
#
# Cookie Demonstration Application
#
# The application requests information from the user, stores the user's 
# name and zip code in cookies, and then displays the information.
#
# Key concepts illustrated include:
#	- Passing in and receiving Cookies
#
# Application Flow:
#
# A user's navigation through the application is represented as state
# machine transitions.  Each URL encodes the state to transition to in
# the "NEXT" (Next State) argument.  The main routine of the application
# examines the NEXT argument and either returns the appropriate static
# WML deck, or invokes the appropriate routine to compute the dynamic
# WML deck.
#
# The NEXT state could be one of the following:
# HOME or blank - To start the application and jump to NEWINFO or SHOWINFO
# NEWINFO  - To retrieve the user's name and zip code
# SHOWINFO - To display the user's name and zip code
#
#
######################################################################

######################################################################
#
# SDK SAMPLE APPLICATION CONFIGURATION
#
# To configure this application to run on your system:
#
# 1. On Unix, change the first line of the file to use the
#    path for the Perl interpreter on your system
#
# 2. On NT, verify that the .cgi file extension is mapped to
#    the Perl interpreter on your system or rename the file
#    extension to the default extension that is mapped to the
#    Perl interpreter on your system (e.g. .pl)
#
######################################################################

######################################################################
# AppUtils Loading and Configuration
######################################################################
# Include the apputils directory which has all the SDK utilities
BEGIN { push (@INC, "../apputils");	} # Path to SDK apputils

# Load required application utilities
require 'DeckUtils.pl';
require 'FileUtils.pl';
require Digest;
require Fax;

######################################################################
# Cookie Retrieval
######################################################################

# Split each cookie pair out of the header string
#
@nvpairs=split(/[,;] */, $ENV{'HTTP_COOKIE'});

# To extract the value from a specific cookie:
#
foreach $pair (@nvpairs) {
   ($name, $value) = split(/=/, $pair);
   $cookie{$name} = $value;
}

######################################################################
# Constants
######################################################################

# The Decks

# Home page for the application
# This card requests input

# Utility deck to display data to the user
$HOME =
'<wml>
    <head>
        <meta forua="true" http-equiv="Cache-Control" content="max-age=0"/>
    </head>
	<card>
		<do type="accept" label="Set">
			<go method="post" href="?NEXT=SHOWINFO">
			<postfield name="NEXT" value="SHOWINFO"/>
			<postfield name="NAME" value="$name"/>
			<postfield name="ZIP" value="$zip"/>
			</go>
		</do>
		<p>
		First name: <input name="name" format="A*a" maxlength="20"/>
		Zip code: <input name="zip" format="NNNNN"/>
		</p>
	</card>
</wml>';

# Displays information returned to the service by the stored cookies
$SHOWINFO =
'<wml>
  <card>
		<do type="options" label="New">
			<go href="?NEXT=NEWINFO"/>
		</do>
	<p>
		Name: %s
		<br/>Zip:  %s
	</p>
  </card>
</wml>';

######################################################################
# State definitions
#
# main uses these definitions to determine next state
######################################################################
# Static states {state, deck}
%StaticStateArray = (
    "STATE", $DECK,
	);

# Dynamic states {state, routine}
%DynamicStateArray = (
	"HOME",     "splashScreen",
	"SHOWINFO", "showInfo",
	"NEWINFO",   "newInfo"
	);


#Call the main function to begin execution
&main;

######################################################################
#
# main Function
#
# When a URL request is received, the application checks the value of
# NEXT to determine the next action. The next action may be to display
# a static WML deck or to execute a function to generate dynamic WML.
# This function implements this logic
#
# This function also reads the CGI variables into %cgiVars for the
# rest of the application to use
#
######################################################################
sub main
{
	# Parse the HTTP CGI variables into the associative array %cgiVars
	%cgiVars = &AppUtils::ParseCGIVars();

	# Get the next state and look it up
	$nextState = $cgiVars{"NEXT"};
	if( !defined($nextState) )	{
		$nextState = "HOME";
	}

	# Transition to the next state
	$deck = $StaticStateArray{$nextState};
	if( $deck ne "" ) {
		#Found a static deck to return for the NEXT state, just return it

		# Create a digest with an WML deck
		my $digest;
		$digest = new Digest;
		$digest->addDeck("", $deck);

		&AppUtils::OutputDigest($digest->asString());
	}
	else {
		$subRoutine = $DynamicStateArray{$nextState};
		if ($subRoutine ne undef) {
			# Found a function to execute for the NEXT state, execute it
			# to generate dynamic WML 
			&$subRoutine();
		}
		else {
			#Invalid NEXT state, return Error deck
			&AppUtils::ErrorExit("Invalid URL", "");
		}
	}
}

######################################################################
# splashScreen
#
# If no cookies have been set, display an input card. If cookies
# exist, display the cookie info. A "new" OPTION will let the user
# enter new cookie info.
#
######################################################################
sub splashScreen
{
	if ($cookie{'name'} ne "") {
		showInfo();
	}
	else {
		newInfo();
	}
}

######################################################################
# showInfo
#
# Show the current Name and Zipcode. If the input was from cgi vars
# then update the cookies by calling addCookie with the new info.
#
######################################################################
sub showInfo
{
	# if cgi vars came in with the request, use them.
	if ($cgiVars{'NAME'} ne "") {
		$name = $cgiVars{"NAME"};
		$zip  = $cgiVars{"ZIP"};
	}
	else {
		$name = $cookie{'name'};
		$zip  = $cookie{'zip'};
	}

	# Create a digest with an WML deck
	my $digest;
	$digest = new Digest;
	$digest->addDeck("", sprintf($SHOWINFO,$name,$zip));

	if ($cgiVars{'NAME'} ne "") {
		# Send the vars back to the browser as cookies. Future requests
		# from the browser will pass these cookies along.
		$digest->addCookie("name=$name,zip=$zip");
	}

	&AppUtils::OutputDigest($digest->asString());
}

######################################################################
# newInfo
#
# Let the user input a name and zip code.
#
######################################################################
sub newInfo
{
	# Return a digest containg the input deck
	#
	my $digest;
	$digest = new Digest;
	$digest->addDeck("", $HOME);
	&AppUtils::OutputDigest($digest->asString());
}

