############################################################################
#
#   Copyright 1999-2000 Phone.com, Inc.  All rights reserved.
#
#   Subject to the terms and conditions of the SDK License Agreement, 
#   Phone.com, Inc. hereby grants you the right to use the UP.SDK 
#   software and its related documentation.
#
#   The Phone.com name and logo and the family of terms carrying the "UP."
#   prefix are trademarks of Phone.com, Inc. All other brands and product 
#   names may be trademarks of the respective companies with which they 
#   are associated.
#
############################################################################

######################################################################
#
# Fax.pm
#
# CLASS:
# Fax
#
# INDEX: (lists the categories this function is indexed under in the docs)
# faxes:sending
# 
# DESCRIPTION:
# The Fax class provides an API to build a facsimile request for the
# UP.Link Fax manager.
#
# SUPERCLASSES:
# MultipartMsg
#
# EXAMPLE:
# require Fax; 
# require Digest;
# ...
# my($faxNum) = "+1 415 596 0805";
# my($fax) = new Fax($faxNum, $TESTFAX, "text/plain");
# $fax->setRecipientName("the Recipient");
# ...
# my($digest) = new Digest;
# $digest->addFax($fax);
# $digest->addDeck("", '<wml><card><p>Fax sent to $faxNum</p></card></wml>');
# ...
# &AppUtils::OutputDigest($response->asString());
#
######################################################################

# Require Perl 5.001 to run
require 5.001;

require Mime;

package Fax;

# for non-unix platforms so they properly calculate msg size.
binmode(STDOUT);

# Specify inheritance
@ISA = ("MultipartMsg");
@Fax::SUPER::ISA = @ISA;

######################################################################
#
# METHOD:
# new
#
# INDEX: (lists the categories this function is indexed under in the docs)
# faxes:creating
#
# DESCRIPTION:
# Creates a new Fax.  Calling this method is a prerequisite for
# calling any other method of this class.
#
# SYNOPSIS:
# require Fax; 
# require Digest;
# ...
# $fax = Fax->new($faxNumber, $faxContent, $contentType);
#
# ARGUMENTS:
# $faxNumber	(string) Fax number of recipient
# $faxContent	(string) Content of the facsimile
# $contentType	(string) MIME content type of the facsimile
#
# RETURNS:
# $self	Returns a reference to the Fax object on success	
#
# EXAMPLE:
# $fax = Fax->new("+1 800 555 1212", $faxContent, "text/plain");
#
######################################################################
sub new
{
	my($class) = shift;
	my($faxNumber) = shift;
	my($content) = shift;
	my($contentType) = shift;
	my($self);

	# Sanity check
	if (!defined($faxNumber)) {
		die "Fax->new(): invalid arguments";
	}

	# Instantiate
	$self = MultipartMsg->new({"Content-type" => "multipart/mixed"});

	my $faxParams = 
		new MsgEntity({"Content-type" => "application/remote-printing"});
	$self->appendSubpart($faxParams);

	my $entry;
	if (defined($content) && defined($contentType)) {
		$entry = MsgEntity->new({"Content-type" => $contentType}, $content);
		$self->appendSubpart($entry);
	}

	bless($self, $class);
	
	# Set up the default headers
	$self->{'fax-params'} = $faxParams;
	$self->{'recipient-name'} = "";
	$self->{'recipient-tel'} = "";
	$self->{'recipient-fax'} = $faxNumber;
	$self->{'sender-name'} = "";
	$self->{'sender-tel'} = "";
	$self->{'sender-fax'} = "";
	$self->{'cover-comment'} = "";

	$self->enableCoverPage(1);

	return $self;
}

######################################################################
#
# METHOD:
# addContent
#
# INDEX: (lists the categories this function is indexed under in the docs)
# faxes:adding content to
#
# DESCRIPTION:
# Adds a document to the contents of the fax.
#
# SYNOPSIS:
# require Fax; 
# require Digest;
# ...
# $fax->addContent($faxContent, $contentType);
#
# ARGUMENTS:
# $faxContent	(string) The content to add
# $contentType	(string) MIME content type of the new content
#
# EXAMPLE:
# $fax->addContent($faxContent, "application/postscript");
#
#
######################################################################
sub addContent {
	my($self) = shift;
	my($content) = shift;
	my($contentType) = shift;

	if (defined($content) && defined($contentType)) {
		$self->appendSubpart(
			MsgEntity->new({"Content-type" => $contentType}, $content));
	}
}


######################################################################
#
# METHOD:
# setRecipientName
#
# INDEX: (lists the categories this function is indexed under in the docs)
# faxes:setting recipient name for
#
# DESCRIPTION:
# Sets the name of the fax recipient.
#
# SYNOPSIS:
# require Fax; 
# require Digest;
# ...
# $fax->setRecipientName($name);
#
# ARGUMENTS:
# $name			(string) The name of the recipient
#
# EXAMPLE:
# $fax->setRecipientName("John Doe");
#
######################################################################
sub setRecipientName
{
	my($self) = shift;
	my($name) = shift;

	# Sanity check
	if (! defined($name)) {
		die "Fax->setRecipientName(): invalid arguments";
	}

	$self->{'recipient-name'} = $name;
}


######################################################################
#
# METHOD:
# setRecipientFax
#
# INDEX: (lists the categories this function is indexed under in the docs)
# faxes:setting recipient fax number for
#
# DESCRIPTION:
# Sets the fax number of the recipient.
#
# require Fax; 
# require Digest;
# ...
# $fax->setRecipientFax($fax);
#
# ARGUMENTS:
# $fax			(string) The fax number of the recipient
#
# EXAMPLE:
# $fax->setRecipientFax("+1 800 555 1212");
#
######################################################################
sub setRecipientFax
{
	my($self) = shift;
	my($fax) = shift;

	# Sanity check
	if (! defined($fax)) {
		die "Fax->setRecipientFax(): invalid arguments";
	}

	$self->{'recipient-fax'} = $name;
}


######################################################################
#
# METHOD:
# setRecipientPhone
#
# INDEX: (lists the categories this function is indexed under in the docs)
# faxes:setting recipient phone number for
#
# DESCRIPTION:
# Sets the phone number of the recipient.
#
# SYNOPSIS:
# require Fax; 
# require Digest;
# ...
# $fax->setRecipientPhone($phone);
#
# ARGUMENTS:
# $phone			(string) The phone number of the recipient
#
# EXAMPLE:
# $fax->setRecipientPhone("+1 800 555 1212");
#
######################################################################
sub setRecipientPhone
{
	my($self) = shift;
	my($phone) = shift;

	# Sanity check
	if (! defined($phone)) {
		die "Fax->setRecipientPhone(): invalid arguments";
	}

	$self->{'recipient-tel'} = $name;
}


######################################################################
#
# METHOD:
# setSenderName
#
# INDEX: (lists the categories this function is indexed under in the docs)
# faxes:adding content to
#
# DESCRIPTION:
# Sets the name of the sender
#
# require Fax; 
# require Digest;
# ...
# $fax->setSenderName($name);
#
# ARGUMENTS:
# $name			(string) The name of the sender
#
# EXAMPLE:
# $fax->setSenderName("John Doe");
#
######################################################################
sub setSenderName
{
	my($self) = shift;
	my($name) = shift;

	# Sanity check
	if (! defined($name)) {
		die "Fax->setSenderName(): invalid arguments";
	}

	$self->{'sender-name'} = $name;
}


######################################################################
#
# METHOD:
# setSenderFax
#
# INDEX: (lists the categories this function is indexed under in the docs)
# faxes:setting sender fax number for
#
# DESCRIPTION:
# Sets the fax number of the sender.
#
# SYNOPSIS:
# require Fax; 
# require Digest;
# ...
# $fax->setSenderFax($fax);
#
# ARGUMENTS:
# $fax			(string) The fax number of the sender
#
# EXAMPLE:
# $fax->setSenderFax("+1 800 555 1212");
#
######################################################################
sub setSenderFax
{
	my($self) = shift;
	my($fax) = shift;

	# Sanity check
	if (! defined($fax)) {
		die "Fax->setSenderFax(): invalid arguments";
	}

	$self->{'sender-fax'} = $name;
}


######################################################################
#
# METHOD:
# setSenderPhone
#
# INDEX: (lists the categories this function is indexed under in the docs)
# faxes:setting sender phone number for
#
# DESCRIPTION:
# Sets the phone number of the sender.
#
# SYNOPSIS:
# require Fax; 
# require Digest;
# ...
# $phone->setSenderPhone($phone);
#
# ARGUMENTS:
# $phone			(string) The phone number of the sender
#
# EXAMPLE:
# $phone->setSenderPhone("+1 800 555 1212");
#
######################################################################
sub setSenderPhone
{
	my($self) = shift;
	my($phone) = shift;

	# Sanity check
	if (! defined($phone)) {
		die "Phone->setSenderPhone(): invalid arguments";
	}

	$self->{'sender-tel'} = $name;
}


######################################################################
#
# METHOD:
# setComment
#
# INDEX: (lists the categories this function is indexed under in the docs)
# faxes:comments, adding
#
# DESCRIPTION:
# Sets the comment on the cover page.
#
# SYNOPSIS:
# require Fax; 
# require Digest;
# ...
# $fax->setSenderFax($comment);
#
# ARGUMENTS:
# $comment		(string) The comment
#
# EXAMPLE:
# $fax->setSenderFax("This is the cool Perl code I told you about");
#
######################################################################
sub setComment
{
	my($self) = shift;
	my($comment) = shift;

	# Sanity check
	if (! defined($comment)) {
		die "Fax->setSenderFax(): invalid arguments";
	}

	# Force newlines to be valid header continuations
	$comment =~ s/\n/\n\t/g;
	$self->{'cover-comment'} = $comment;
}


######################################################################
#
# METHOD:
# enableCoverPage
#
# INDEX: (lists the categories this function is indexed under in the docs)
# faxes:cover page, enabling
#
# DESCRIPTION:
# Specifies whether a cover page is attached to the fax or not.
#
# SYNOPSIS:
# require Fax; 
# require Digest;
# ...
# $fax->enableCoverPage($addCover);
#
# ARGUMENTS:
# $addCover		(boolean) Include cover page (or not)
#
# EXAMPLE:
# $fax->enableCoverPage(1);
#
######################################################################
sub enableCoverPage
{
	my($self) = shift;
	my($enableCover) = shift;

	my $faxParams = $self->{'fax-params'};
	if ($enableCover) {
		$faxParams->addHeader("X-up-fax-cover", "true");
	} else {
		$faxParams->addHeader("X-up-fax-cover", "false");
	}
}

######################################################################
#
# METHOD:
# asString
#
# INDEX: (lists the categories this function is indexed under in the docs)
# faxes:converting to string
#
# DESCRIPTION:
# Converts the multipart message to a string for printing.
#
# SYNOPSIS:
# require Fax; 
# require Digest;
# ...
# $content = $fax->asString();
#
# RETURN:
# $content		The entire fax message as a single scalar string
#
# EXAMPLE:
# $content = $fax->asString();
#
# CAVEAT:
# On non-Unix systems, you should call "binmode" on a file descriptor
# before writing this string out to it
#
######################################################################
sub asString
{
	my ($self) = shift;

	my $faxParams = $self->{'fax-params'};
	my ($faxParamsBody);

	#Build the body of the fax parameters sub part based using the headers
	#that have been specified
	$body  = "Recipient: ".$self->{'recipient-name'}."\n";
	$body .= "Telephone: ".$self->{'recipient-tel'}."\n";
	$body .= "Facsimile: ".$self->{'recipient-fax'}."\n";
	# XXX - should be CRLF not LF
	$body .= "\n";

	$body .= "Originator: ".$self->{'sender-name'}."\n";
	$body .= "Telephone: ".$self->{'sender-tel'}."\n";
	$body .= "Facsimile: ".$self->{'sender-fax'}."\n";
	# XXX - should be CRLF not LF
	$body .= "\n";

	$body .= $self->{'cover-comment'};
	$body .= "\n";
	$faxParams->setBody($body);

	#At this point the fax parameters sub part is fully constructed
	#Any fax content should be available as sub parts in this object
	#We can now call MultipartMsg::asString to build the output string
	return $self->Fax::SUPER::asString();
}


