
#define PMWHO           "pmwho"
#define PMWHO_DEFAULT   "pm1"
#define PASSWORD        "password"
struct {
  char hostname[32];
  char password[32];
} pwtable[] = {
  /* { "pm1", "topsecret" } */
};

#if defined(SUNOS4) || defined(SOLARIS)
#define TELNET_EXEC     "/usr/ucb/telnet"
#define TELNET_CLOSE    "Connection closed"
#define TELNET_PORT     "23"
#define TELNET_PROMPT   "telnet> "
#define HAVE_SYS_WAIT
#elif defined(DNIX) || defined(HPUX)
#define TELNET_EXEC     "/usr/bin/telnet"
#define TELNET_CLOSE    "Trying"
#define TELNET_PORT     "23"
#define TELNET_PROMPT   "telnet> "
#define HAVE_SYS_WAIT
#elif defined(LINUX)
#define TELNET_EXEC     "/bin/telnet"
#define TELNET_CLOSE    "Connection closed"
#define TELNET_PORT     "23"
#define TELNET_PROMPT   "telnet> "
#define HAVE_SYS_WAIT
#else /* generic entry */
#define TELNET_EXEC     "/usr/ucb/telnet"
#define TELNET_CLOSE    "Connection closed"
#define TELNET_PORT     "23"
#define TELNET_PROMPT   "telnet> "
/* #define HAVE_SYS_WAIT */
/* #define HAVE_WAIT */
#endif

/* header files */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#if defined(HPUX)
#include <unistd.h>
#define NSIG _NSIG
#endif

#include <sys/types.h>

#if defined(HAVE_SYS_WAIT)
#include <sys/wait.h>
#elif defined(HAVE_WAIT)
#include <wait.h>
#else
extern pid_t wait(int *status);
#endif

#include <errno.h>
#include <signal.h>

char  *progname;
char  *hostname;
pid_t pid;
int   showall;
int   got_alarm;

#define LINELEN  256
#define IDLE_POS  50
#define kas
#define mas
#define hey
void main(int argc, char *argv[]);
void usage(void);
void pmwho(void);
int  do_read(int fd, char *text1, char *text2, int show, int timeout);
int  do_write(int fd, char *text, int timeout);
void sigalrm(int signo);
void sigfunc(int signo);
void err_msg(const char *format, ...);
void err_exit(const char *format, ...);
void stop_telnet(void);

/* the program */

/* main procedure, find out program name and the hostname to use */
void main(int argc, char *argv[])
{
  char *t;
  int ch, n;

  extern int optind;

  t = strrchr(argv[0], '/');
  progname = (t != NULL) ? t+1 : argv[0];
  hostname = NULL;
  pid = -1;
  showall = 0;

  while ((ch = getopt(argc, argv, "ah")) != -1) {
    switch (ch) {
      case 'a':
        showall = 1;
        break;
      case 'h':
        /* fall-through */
      default:
        usage();
    }
  }

  /* setup signals properly */
  for(n = 1; n < NSIG; n++) signal(n, sigfunc);
#if defined(SIGCHLD)
  signal(SIGCHLD, SIG_IGN);
#endif
#if defined(SIGCLD)
  signal(SIGCLD, SIG_IGN);
#endif

  /* take different action according to usage */
  if (strcmp(progname, PMWHO) == 0) { /* PMWHO */
    if (optind >= argc) { /* usage: 'PMWHO' */
      hostname = PMWHO_DEFAULT;
      pmwho();
    }
    else if (optind == argc-1) { /* usage: 'PMWHO hostname' */
      hostname = argv[optind];
      pmwho();
    }
    else { /* usage: 'PMWHO hostname1 hostname2 ...' */
      for(; optind < argc; optind++) {
        hostname = argv[optind];
        fprintf(stdout, "[%s]\n", hostname);
        fflush(stdout);
        pmwho();
        if (optind < argc-1) {
          fprintf(stdout, "\n");
          fflush(stdout);
        }
      }
    }
  }
  else { /* usage: 'hostname' */
    hostname = progname;
    progname = PMWHO; /* for the messages */
    pmwho();
  }
  exit(0);
}

/* show usage info */
void usage(void)
{
  fprintf(stderr, "usage: %s [-ah]\n", progname);
  exit(0);
}

/* retrieve the information from the host by logging on, giving the command */
/* "show session" and logging out from 'hostname'                           */
void pmwho()
{
  int fdpipe[2][2], fdin, fdout, fderr, err, n;

  got_alarm = 0;

  /* we need pipes to communicate between the programs */
  if (pipe(fdpipe[0]) < 0) {
    err_msg("%s: pipe failed: errno=%d\n", progname, errno);
    return;
  }
  if (pipe(fdpipe[1]) < 0) {
    err_msg("%s: pipe failed: errno=%d\n", progname, errno);
    close(fdpipe[0][0]);
    close(fdpipe[0][1]);
    return;
  }

  switch(pid = fork()) {
    case 0: /* child */
      /* child:stdin */
      close(0);
      if (dup(fdpipe[0][0]) < 0) {
        err_exit("%s: dup failed: errno=%d\n", progname, errno);
      }

      /* close first pipe */
      close(fdpipe[0][0]);
      close(fdpipe[0][1]);

      /* child:stdout */
      close(1);
      if (dup(fdpipe[1][1]) < 0) {
        err_exit("%s: dup failed: errno=%d\n", progname, errno);
      }

      /* child:stderr */
      if ((fderr = dup(2)) < 0) {
        err_exit("%s: dup failed: errno=%d\n", progname, errno);
      }

      close(2);
      if (dup(fdpipe[1][1]) < 0) {
        err = errno;
        dup(fderr);
        err_exit("%s: dup failed: errno=%d\n", progname, err);
      }

      /* close second pipe */
      close(fdpipe[1][0]);
      close(fdpipe[1][1]);

      /* exec TELNET application */
      execl(TELNET_EXEC, "telnet", hostname, TELNET_PORT, (char*)NULL);

      /* if we're still here the TELNET_EXEC could not be exec'd */
      err = errno;
      close(2);
      dup(fderr);
      err_exit("%s: execl(%s) failed: errno=%d\n", progname, TELNET_EXEC, err);
      break;

    case -1: /* error */
      err_exit("%s: fork failed: errno=%d\n", progname, errno);
      return;

    default: /* parent */
      /* close the childs end of the pipes */
      close(fdpipe[0][0]);
      close(fdpipe[1][1]);
      break;
  }

  /* now communicate with the 'telnet' process */
  fdin = fdpipe[1][0];
  fdout = fdpipe[0][1];

  for(;;) {
    n = do_read(fdin, "login: ", TELNET_PROMPT, 0, 15);
    if (n != 1) {
      n = -1; /* TELNET_PROMPT == connection failed */
      break;
    }
    if ((n = do_write(fdout, "!root\n", 5)) < 0) break;

    if ((n = do_read(fdin, "Password: ", NULL, 0, 10)) < 0) break;

    {
      char *pw = PASSWORD;
      int k;

      for(k = 0; k < sizeof(pwtable)/sizeof(pwtable[0]); k++) {
        if (strcmp(pwtable[k].hostname, hostname) == 0) {
          pw = pwtable[k].password;
          break;
        }
      }

      if ((n = do_write(fdout, pw, 5)) < 0) break;
    }

    if ((n = do_write(fdout, "\n", 5)) < 0) break;

    if ((n = do_read(fdin, "> ", NULL, 0, 10)) < 0) break;
    if ((n = do_write(fdout, "show session\n", 5)) < 0) break;

    for(;;) {
      n = do_read(fdin, "-- Press Return for More -- ", "> ", 1, 20);
      if ((n < 0) || (n == 2)) break;
      if ((n = do_write(fdout, "\n", 5)) < 0) break;
    }
    if (n < 0) break;

    if ((n = do_write(fdout, "quit\n", 5)) < 0) break;
    
    if ((n = do_read(fdin, TELNET_CLOSE, NULL, 0, 20)) < 0) break;

    break;
  }

  close(fdin);
  close(fdout);

  if ((n < 0) && (got_alarm == 0)) {
    err_msg("%s: Sorry host is not available '%s' failed\n", progname, hostname);
  }

  stop_telnet();
}

int do_read(int fd, char *text1, char *text2, int show, int timeout)
{
  char line[LINELEN+1];
  int n, len1, len2, count = 0, m, err;

  /* setup alarm (so we won't hang forever upon problems) */
  signal(SIGALRM, sigalrm);
  alarm(timeout);
  
  len1 = strlen(text1);
  len2 = (text2 != NULL) ? strlen(text2) : 0;

  /* start reading from 'telnet' */
  for(;;) {
    n = 0;
    for(;;) {
      if (n == LINELEN) {
        alarm(0); /* disable alarm */
        stop_telnet();
        err_exit("%s: too long line!\n", progname);
      }
      m = read(fd, &line[n], 1);
      if (m != 1) {
        err = errno;
        alarm(0); /* disable alarm */
#if 0
        if (m < 0) {
          err_msg("%s: read failed: errno=%d\n", progname, err);
        }
#endif
        return(-1);
      }
      if ((line[n] == '\r') || (m == 0)) continue;
      if (n >= len1-1) {
        if (strncmp(&line[n-(len1-1)], text1, len1) == 0) {
          /* we found the keyword we were searching for */
          alarm(0); /* disable alarm */
          return(1);
        }
      }
      if ((text2 != NULL) && (n >= len2-1)) {
        if (strncmp(&line[n-(len2-1)], text2, len2) == 0) {
          alarm(0); /* disable alarm */
          return(2);
        }
      }
      if (line[n] == '\n') break;
      n++;
    }
    if (show) {
      line[++n] = '\0';
      if (++count > 1) {
        /* the very first line of information is the remains of  */
        /* a previously issued command and should be ignored.    */

        if ((line[0] == 'P') && (line[1] >= '0') && (line[1] <= '9')) {
          /* parallel port info */
#if defined(SHOW_PRINTER)
            fputs(line, stdout);
            fflush(stdout);
#endif
        }
        else {
          /* serial port info */
          if (showall) {
            fputs(line, stdout);
            fflush(stdout);
          }
          else if (strlen(line) >= IDLE_POS) {
            if (strncmp(&line[IDLE_POS], "IDLE ", 5) != 0) {
              fputs(line, stdout);
              fflush(stdout);
            }
          }
        }
      }
    }
  }
}

/* write a command to the 'telnet' process */
int do_write(int fd, char *text, int timeout)
{
  int err, len, n;

  /* setup alarm (so we won't hang forever upon problems) */
  signal(SIGALRM, sigalrm);
  alarm(timeout);

  len = strlen(text);
  n = write(fd, text, len);
  if (n != len) {
    err = errno;
    alarm(0); /* disable alarm */
#if 0
    if (n < 0) {
      err_msg("%s: write failed: errno=%d\n", progname, err);
    }
#endif
    return(-1);
  }
  else {
    alarm(0); /* disable alarm */
    return(0);
  }
}

/* our timeout procedure, used to abort malfunctioning connections */
void sigalrm(int signo)
{
  got_alarm = 1;
  err_msg("%s: timeout on connection to host '%s'\n", progname, hostname);
}

/* handle the reception of signals */
void sigfunc(int signo)
{
#if defined(SIGPIPE)
  if (signo != SIGPIPE)
#endif
  {
    err_msg("%s: received signal #%d during connection to host '%s' -- aborting\n", progname, signo, hostname);
  }

  stop_telnet();
  exit(1);
}

/* print error text */
void err_msg(const char *format, ...)
{
  va_list ap;

  /* show error message */
  va_start(ap, format);
  vfprintf(stderr, format, ap);
  va_end(ap);
  fflush(stderr);
}

void err_exit(const char *format, ...)
{
  va_list ap;

  /* show error message */
  va_start(ap, format);
  vfprintf(stderr, format, ap);
  va_end(ap);
  fflush(stderr);

  /* exit with error */
  exit(1);
}

void stop_telnet(void)
{
  pid_t p;

  if ((pid != -1) && (pid != 0)) {
    if (kill(pid, 0) >= 0) {
      kill(pid, SIGTERM);
      kill(pid, SIGKILL);

      if (kill(pid, 0) >= 0) {
        for(p = wait(0); (p != -1) && (p != pid); p = wait(0)) ; /* zombies */
      }
    }
    pid = -1;
  }
}
