######################################################################
#
# Copyright 1997 Unwired Planet, Inc.  All rights reserved.
#
# Subject to the terms and conditions of the LICENSE AGREEMENT FOR
# SDK SOFTWARE, Unwired Planet, Inc. hereby grants you authorization
# to use UP.SDK software and its related documentation.
#
# Unwired Planet, the Unwired Planet logo, UP.Phone, UP.Link, UP.Organizer
# UP.Browser, UP.Access, UP.Mail, UP.Pager, HDML, UPLANET.COM and
# UP.SDK are trademarks of Unwired Planet, Inc.
#
######################################################################
 
######################################################################
#
# This package contains some routines for accessing the todo data file.
#
# This is a very simple database class for the todo list. The data
# is kept in a text file using the following format
#
# status|description
#
# Finished items (those with a F status) come after all other items.
# The order in the file is important. It should be the same order that
# the todo's are displayed.
#
######################################################################

# Require Perl 5.001 to run
require 5.001;

package TodoData;

# IMPORTANT:  Change this path to the data directory on your web server.
#	      The directory must be writable by the web server process.
#
$dataDir = '/usr/local/apache/cgi-bin/wap';

# New
#
# Input:   SubscriberId
# Output:  None
# Returns: A todo list database object
sub New
{
	my $class = shift;
	my $subId = shift;

	my $todoFile = $dataDir . "/$subId";

	TodoData->CheckDataDir();
	TodoData->CheckDataFile($todoFile);

	bless \$todoFile, $class;
}

# GetActiveItems
#
# Input: none
# Output: none
# Returns: An array of todo items
sub GetActiveItems
{
	my $self = shift;
	my $status;
	my $text;
	my @activeItems;

	unless (open(TODO_FD, "<$$self")) {
		return(@activeItems);
	}

	# Seek to the beginning
	seek(TODO_FD, 0, 0);

	my @fileBuf = <TODO_FD>;

	close(TODO_FD);

	foreach(@fileBuf) {
		($status, $text) = split('\|', $_, 2);

		# If this is an active item, add it to the return list
		if ($status ne 'F') {
			my %item;
			$item{'status'} = $status;
			$item{'text'} = $text;
#			unshift(@activeItems, \%item);
			push(@activeItems, \%item);
		}
	}

	@activeItems;
}

# GetFinishedItems
#
# Input: none
# Output: none
# Returns: An array of todo items
sub GetFinishedItems
{
	my $self = shift;
	my $status;
	my $text;
	my @finishedItems;

	unless (open(TODO_FD, "<$$self")) {
		return(@activeItems);
	}

	# Seek to the beginning
	seek(TODO_FD, 0, 0);

	my @fileBuf = <TODO_FD>;

	close(TODO_FD);

	foreach(@fileBuf) {
		($status, $text) = split('\|', $_, 2);

		# If this is a finished item, add it to the return list
		if ($status eq 'F') {
			my %item;
			$item{'status'} = $status;
			$item{'text'} = $text;
			push(@finishedItems, \%item);
		}
	}

	@finishedItems;
}

# InsertNewItem
#
# Input: Todo item
# Output: none
# Return: none
sub InsertNewItem
{
	my $self = shift;
	my $todoItem = shift;

	unless (open(TODO_FD, "+<$$self")) {
		return;
	}

	# Seek to the beginning
	seek(TODO_FD, 0, 0);

	# Read in the lines
	my @fileBuf = <TODO_FD>;

	# Truncate the file
	seek(TODO_FD, 0, 0);
	truncate(TODO_FD, 0);

	# Write the file with the new addition
	print TODO_FD "N|$todoItem\n", @fileBuf;

	close(TODO_FD);
}

# SetStatus
#
# Input: Index of item to modify, status
# Output: none
# Return: none
sub SetStatus
{
	my $self = shift;
	my $index = shift;
	my $newStatus = shift;
	my @activeItems;
	my @finishedItems;
	my $status;
	my $text;

	unless (open(TODO_FD, "+<$$self")) {
		return;
	}

	# Index on the phone is 1 based, make it 0 based
	$index -= 1 if ($index > 0);

	# Seek to the beginning
	seek(TODO_FD, 0, 0);

	my @fileBuf = <TODO_FD>;

	foreach(@fileBuf) {
		($status, $text) = split('\|', $_, 2);

		# Add the line to the correct array
		if ($status ne 'F') {
			push(@activeItems, ($_));
		}
		else {
			push(@finishedItems, ($_));
		}
	}

	($status, $text) = split('\|', $activeItems[$index], 2);
	if ($newStatus ne 'F') {
		# update the status in the active array
		$activeItems[$index] = "$newStatus|$text";
	}
	else {
		# Remove it from the active array and place it in the finished array
		splice(@activeItems, $index, 1);
		push(@finishedItems, ("F|$text"));
	}

	# Truncate the file
	seek(TODO_FD, 0, 0);
	truncate(TODO_FD, 0);

	# Output the active items and then the finished items
	print TODO_FD @activeItems, @finishedItems;

	close(TODO_FD);
}

# DeleteFinishedItems
#
# Input: none
# Output: none
# Return: none
sub DeleteFinishedItems
{
	my $self = shift;
	my @activeItems;
	my $status;
	my $text;

	unless (open(TODO_FD, "+<$$self")) {
		return;
	}

	# Seek to the beginning
	seek(TODO_FD, 0, 0);

	my @fileBuf = <TODO_FD>;

	foreach(@fileBuf) {
		($status, $text) = split('\|', $_, 2);

		# Add the line to the correct array
		if ($status ne 'F') {
			push(@activeItems, ($_));
		}
	}

	# Truncate the file
	seek(TODO_FD, 0, 0);
	truncate(TODO_FD, 0);

	# Output the active items
	print TODO_FD @activeItems;

	close(TODO_FD);
}


# Static member function
#
# Check to see if the data dir exists. If not, create it.
# Also check to see if it is writeable.
#
# Return 1 if everything is OK, 0 otherwise
sub CheckDataDir
{
	# Does it exist
	if (! -e $dataDir) {
		# Directory did not exist, create it
		unless (mkdir($dataDir, 0777)) {
			return(0);
		}
	}

	# It exists, is it a directory?
	if (! -d $dataDir) {
		return(0);
	}

	# Is it writable by us?
	if (! -w $dataDir || ! -r $dataDir) {
		return(0);
	}

	1;
}

# Static member function
#
# Check to see if the data file exists. If not, create it.
#
# Return 1 if everything is OK, 0 otherwise
sub CheckDataFile
{
	my $class    = shift;
	my $dataFile = shift;

	# Does it exist
	if (! -e $dataFile) {
		# Directory did not exist, create it
		unless (open(TODO_FD, ">$dataFile")) {
			return(0);
		}
		close(TODO_FD);
	}

	# Is it writable and readable by us?
	if (! -w $dataFile || ! -r $dataFile) {
		return(0);
	}

	1;
}

1;
