package ejb;

import util.*;
import security.*;
import java.util.*;
import javax.ejb.*;
import javax.naming.*;
import com.caucho.ejb.*;
import com.caucho.xml.*;
import org.w3c.dom.*;
import javax.servlet.http.*;

public class ForumCtrlBean extends AbstractSessionBean {

  private static final String STATUS_NORMAL = "normal";
  private static final String STATUS_CLOSED = "closed";
  private static final String STATUS_NEW = "new";

  public void ejbCreate() {}

  public Forum createForum(HttpServletRequest request) throws ControlException {

    String name = TextProcessor.normStr(request.getParameter("name"));
    String description = TextProcessor.normStr(request.getParameter("description"));
    if ("".equals(name))
      throw new ControlException("forum name required");
    try {
      Forum forum = null;
      try {
        forum = EJBGetter.getForumHome().findByName(name);
      } catch (FinderException ex) { }
      if (forum != null) 
        throw new ControlException("'" + name + "' is a duplicate forum name");
      forum = EJBGetter.getForumHome().create(name, description);
      return forum;
    } catch (CreateException ex) {
      throw new ControlException(ex.getMessage(), ex);
    }
  }

  public Forum updateForum(HttpServletRequest request) throws ControlException {
    int id;
    try {
      id = Integer.parseInt(request.getParameter("forumId"));
    } catch (Exception ex) {
      throw new ControlException("invalid or missing forum id");
    }
    String name = TextProcessor.normStr(request.getParameter("name"));
    String description = TextProcessor.normStr(request.getParameter("description"));
    if ("".equals(name))
      throw new ControlException("forum name required");
    try {
      Forum forum = null;
      try {
        forum = EJBGetter.getForumHome().findByName(name);
      } catch (FinderException ex) { }
      if ((forum != null) && (forum.getId() != id))
        throw new ControlException("'" + name + "' is a duplicate forum name");
      forum = EJBGetter.getForumHome().findByPrimaryKey(id);
      if (forum.getCurrentPrivilege() < Privilege.MODERATE)
        throw new ControlException("operation not alloved");
      forum.setName(name);
      forum.setDescription(description);
      return forum;
    } catch (FinderException ex) {
      throw new ControlException("forum not found", ex);
    }
  }

  public Topic createTopic(HttpServletRequest request) throws ControlException {
    Forum forum = null;
    Topic parentTopic = null;
    try {
      int parentTopicId = Integer.parseInt(request.getParameter("topicId"));
      parentTopic = EJBGetter.getTopicHome().findByPrimaryKey(parentTopicId);
      forum = parentTopic.getForum();
      if (!parentTopic.getActive())
        throw new ControlException("could not post reply because topic is closed");
    } catch (Exception ex) {
      try {
        int forumId = Integer.parseInt(request.getParameter("forumId"));
        forum = EJBGetter.getForumHome().findByPrimaryKey(forumId);
      } catch (Exception ex1) {
        throw new ControlException("could not create new topic");
      }
    }
    if (!forum.getActive())
      throw new ControlException("could not post new topic because forum is closed");
    if (forum.getCurrentPrivilege() < Privilege.WRITE)
      throw new ControlException("operation not alloved");

    String subject = TextProcessor.normStr(request.getParameter("subject"));
    if ("".equals(subject))
      throw new ControlException("subject required");
    String text = TextProcessor.normStr(request.getParameter("text"));
    text = TextProcessor.preProcess(text);

    if (request.getUserPrincipal() == null)
      throw new ControlException("topic creation is not permited for Anonymous user");
    User user = (User)request.getUserPrincipal();

    Topic ret;
    try {
      if (parentTopic == null)
        ret = EJBGetter.getTopicHome().create(forum, user, subject, text);
      else
        ret = EJBGetter.getTopicHome().create(parentTopic, user, subject, text);
    } catch (CreateException ex) {
      throw new ControlException(ex.getMessage(), ex);
    }
    
    Iterator iter = forum.getPrivileges().iterator();
    while (iter.hasNext()) {
      UserForumPrivilege priv = (UserForumPrivilege)iter.next();
      if ((priv.getPrivilege() >= Privilege.READ) && (!priv.getUser().equals(ret.getUser())) && (priv.getUser().getNotify())) {
        try {
          MailSender mail = EJBGetter.getMailSenderHome().create();
          mail.addTo(priv.getUser().getEmail());
          mail.setFrom(System.getProperty("mail.support"));
          mail.setSubject("FORUM: " + ret.getSubject());
          StringBuffer body = new StringBuffer();
          body.append("Forum: " + ret.getForum().getName() +"\n")
              .append("Author: " + ret.getUser().getFullNameExt() +"\n")
              .append("Subject: " + ret.getSubject() +"\n")
              .append("URL: " + System.getProperty("site.url") + "forum_intopic.jsp?topicId=" + ret.getId() + "\n");
          mail.setBodyText(body.toString());
          mail.send();
          mail.remove();
        } catch (Throwable ex) {
          ex.printStackTrace();
        }
      }
    }
    return ret;
  }

  public Topic updateTopic(HttpServletRequest request) throws ControlException {
    int id;
    try {
      id = Integer.parseInt(request.getParameter("topicId"));
    } catch (Exception ex) {
      throw new ControlException("invalid or missing topic id");
    }
    String subject = TextProcessor.normStr(request.getParameter("subject"));
    if ("".equals(subject))
      throw new ControlException("subject required");
    String text = TextProcessor.normStr(request.getParameter("text"));
    text = TextProcessor.preProcess(text);
    Topic topic;
    try {
      topic = EJBGetter.getTopicHome().findByPrimaryKey(id);
    } catch (FinderException ex) {
      throw new ControlException("topic not found", ex);
    }
    if (topic.getForum().getCurrentPrivilege() < Privilege.WRITE)
      throw new ControlException("operation not alloved");
    topic.setSubject(subject);
    topic.setText(text);
    return topic;
  }

  public Document forumsList() {
    Document doc = Xml.createDocument();
    try {
      Node root = doc.createElement("forums");
      doc.appendChild(root);
      Iterator iter = EJBGetter.getForumHome().findAll().iterator();
      TopicHome topicHome = EJBGetter.getTopicHome();
      while (iter.hasNext()) {
        Forum f = (Forum)iter.next();
        if (f.getCurrentPrivilege() >= Privilege.READ) {
          boolean active = f.getActive();
          int tCnt = f.getTopicsCount();
          int mCnt = f.getMessagesCount();
          Element el = doc.createElement("forum");
          root.appendChild(el);
          el.setAttribute("id", f.getId() + "");
          el.setAttribute("name", f.getName());
          el.setAttribute("description", f.getDescription());
          el.setAttribute("last-post", util.Format.dateTime(f.getLastPost()));
          el.setAttribute("status", "active");
          el.setAttribute("topics", tCnt +  "");
          el.setAttribute("messages", mCnt + "");
        }
      }
    } catch (Exception e) {
    }
    return doc;
  }

  private void collectSubtopics(Node node, Topic parent, Topic current) throws Exception {
    if (parent.getForum().getCurrentPrivilege() < Privilege.READ)
      return;
    Document doc = node.getOwnerDocument();
    Element el = doc.createElement("thread-topic");
    node.appendChild(el);
    el.setAttribute("id", parent.getId() + "");
    el.setAttribute("posted", util.Format.dateTime(parent.getPosted()));
    el.setAttribute("author", parent.getUser().getFullNameExt());
    el.setAttribute("subject", parent.getSubject());
    el.setAttribute("viewed", parent.getViewed() + "");
    el.setAttribute("status", getTopicStatus(parent));
    if (current.equals(parent))
      el.setAttribute("current", "true");
    Iterator iter = parent.getChildren().iterator();
    while (iter.hasNext()) {
      Topic topic = (Topic)iter.next();
      collectSubtopics(el, topic, current);
    }
  }

  public Document threadData(int topicId) throws Exception {
    Topic current = EJBGetter.getTopicHome().findByPrimaryKey(topicId);
    Topic root = current.getRoot();
    Document doc = Xml.createDocument();
    Node node = doc.createElement("thread");
    doc.appendChild(node);
    if (root.getForum().getCurrentPrivilege() >= Privilege.READ)
      collectSubtopics(node, root, current);
    return doc;
  }

  public Document topicData(int topicId) throws Exception {
    Document doc = Xml.createDocument();
    Topic topic = EJBGetter.getTopicHome().findByPrimaryKey(topicId);
    if (topic.getForum().getCurrentPrivilege() >= Privilege.READ) {
      Topic rootTopic = topic.getRoot();
      int msgCnt = topic.getSubtopicsCount();
      Element el = doc.createElement("topic");
      doc.appendChild(el);
      el.setAttribute("id", topic.getId() + "");
      el.setAttribute("posted", util.Format.dateTime(topic.getPosted()));
      el.setAttribute("author", topic.getUser().getFullNameExt());
      el.setAttribute("subject", topic.getSubject());
      el.setAttribute("messages", msgCnt + "");
      el.setAttribute("viewed", topic.getViewed() + "");
      el.setAttribute("status", getTopicStatus(topic));
      Iterator iter = topic.getAttachments().iterator();
      if (iter.hasNext()) {
        Element att = doc.createElement("attachments");
        el.appendChild(att);
        while (iter.hasNext()) {
          Attachment a = (Attachment)iter.next();
          Element file = doc.createElement("file");
          att.appendChild(file);
          file.setAttribute("name", a.getName());
          file.setAttribute("url", "/files/" + topic.getId() + "/" + a.getName());
          file.setAttribute("size", Format.fileSize(a.getSize()));
        }
      }
      el.appendChild(doc.createCDATASection(TextProcessor.viewProcess(topic.getText())));
    }
    return doc;
  }

  public Document topicsList(int forumId) {
    Document doc = Xml.createDocument();
    try {
      Node root = doc.createElement("topics");
      doc.appendChild(root);
      Forum forum = EJBGetter.getForumHome().findByPrimaryKey(forumId);
      if (forum.getCurrentPrivilege() >= Privilege.READ) {
        Iterator iter = EJBGetter.getTopicHome().findRoot(forum).iterator();
        while (iter.hasNext()) {
          Topic topic = (Topic)iter.next();
          boolean active = topic.getActive() && topic.getForum().getActive();
          int msgCnt = topic.getSubtopicsCount();
          Element el = doc.createElement("topic");
          root.appendChild(el);
          el.setAttribute("id", topic.getId() + "");
          el.setAttribute("posted", util.Format.dateTime(topic.getPosted()));
          Topic lastReply = topic.getLastReply();
          if (!lastReply.equals(topic)) {
            el.setAttribute("last-reply-date", util.Format.dateTime(lastReply.getPosted()));
            el.setAttribute("last-reply-author", lastReply.getUser().getFullNameExt());
          }
          el.setAttribute("author", topic.getUser().getFullNameExt());
          el.setAttribute("subject", topic.getSubject());
          el.setAttribute("messages", msgCnt + "");
          el.setAttribute("viewed", topic.getViewed() + "");
          el.setAttribute("status", getThreadStatus(topic));
        }
      }
    } catch (Exception e) {
    }
    return doc;
  }

  public Document searchResult(String text) {
    StringBuffer ret = new StringBuffer();
    Document doc = Xml.createDocument();
    try {
      Node root = doc.createElement("topics");
      doc.appendChild(root);
      Iterator iter = EJBGetter.getTopicHome().findBySubectLike(text).iterator();
      while (iter.hasNext()) {
        Topic topic = (Topic)iter.next();
        if (topic.getForum().getCurrentPrivilege() >= Privilege.READ) {
          boolean active = topic.getActive() && topic.getForum().getActive();
          int msgCnt = topic.getSubtopicsCount();
          Element el = doc.createElement("topic");
          root.appendChild(el);
          el.setAttribute("id", topic.getId() + "");
          el.setAttribute("posted", util.Format.dateTime(topic.getPosted()));
          el.setAttribute("author", topic.getUser().getFullNameExt());
          el.setAttribute("subject", topic.getSubject());
          el.setAttribute("messages", msgCnt + "");
          el.setAttribute("viewed", topic.getViewed() + "");
          el.setAttribute("status", getTopicStatus(topic));
        }
      }
    } catch (Exception e) {
    }
    return doc;
  }

  private String getTopicStatus(Topic topic) {
    boolean active = topic.getActive() && topic.getForum().getActive();
    if (active) {
      User user = (User)getSessionContext().getCallerPrincipal();
      if (user == null)
        return STATUS_NORMAL;
      else if ((user.getPrevLogin() == null) || (user.getPrevLogin().getTime() < topic.getPosted().getTime()))
        return STATUS_NEW;
      else
        return STATUS_NORMAL;
    } else {
      return STATUS_CLOSED;
    }
  }

  private String getThreadStatus(Topic topic) {
    boolean active = topic.getActive() && topic.getForum().getActive();
    if (active) {
      User user = (User)getSessionContext().getCallerPrincipal();
      if (user == null)
        return STATUS_NORMAL;
      else if ((user.getPrevLogin() == null) || (user.getPrevLogin().getTime() < topic.getLastReply().getPosted().getTime()))
        return STATUS_NEW;
      else
        return STATUS_NORMAL;
    } else {
      return STATUS_CLOSED;
    }
  }

}
