/*
 * Copyright (c) 1999-2000 Caucho Technology.  All rights reserved.
 *
 * Caucho Technology permits redistribution, modification and use
 * of this file in source and binary form ("the Software") under the
 * Caucho Public License ("the License").  In particular, the following
 * conditions must be met:
 *
 * 1. Each copy or derived work of the Software must preserve the copyright
 *    notice and this notice unmodified.
 *
 * 2. Redistributions of the Software in source or binary form must include 
 *    an unmodified copy of the License, normally in a plain ASCII text
 *
 * 3. The names "Resin" or "Caucho" are trademarks of Caucho Technology and
 *    may not be used to endorse products derived from this software.
 *    "Resin" or "Caucho" may not appear in the names of products derived
 *    from this software.
 *
 * 4. Caucho Technology requests that attribution be given to Resin
 *    in any manner possible.  We suggest using the "Resin Powered"
 *    button or creating a "powered by Resin(tm)" link to
 *    http://www.caucho.com for each page served by Resin.
 *
 * This Software is provided "AS IS," without a warranty of any kind. 
 * ALL EXPRESS OR IMPLIED REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.

 * CAUCHO TECHNOLOGY AND ITS LICENSORS SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE OR ANY THIRD PARTY AS A RESULT OF USING OR
 * DISTRIBUTING SOFTWARE. IN NO EVENT WILL Caucho OR ITS LICENSORS BE LIABLE
 * FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL,
 * CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND
 * REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
 * INABILITY TO USE SOFTWARE, EVEN IF HE HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGES.      
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <dirent.h>
// probably system-dependent
#include <dlfcn.h>
#include <jni.h>

#include "cse.h"
#include "proxy.h"
#include "proxy_jni.h"

#define OS "i386"

static config_t *g_config;

static int g_server_socket;

typedef struct options_t {
  char *resin_home;
  char *java_home;
  char *conf;
  char *classpath;
  char *loadpath;
} options_t;

static options_t g_options;
java_t g_java;

void
cse_log(char *fmt, ...)
{
}

void *
cse_malloc(int size)
{
  return malloc(size);
}

void
cse_error(config_t *config, char *format, ...)
{
}

void
cse_set_socket_cleanup(int socket, void *pool)
{
}

void
cse_kill_socket_cleanup(int socket, void *pool)
{
}

int
cse_lock(void *lock)
{
  return 1;
}

void
cse_unlock(void *lock)
{
}

void *
cse_create_lock()
{
  return 0;
}

/**
 * Bind to a port.
 *
 * @param hostname name of the host interface
 * @param port port to bind
 *
 * @return server socket on success, -1 on failure.
 */
static int
cse_bind(char *hostname, int port)
{
  struct sockaddr_in sin;
  int val = 0;

  int sock = socket(AF_INET, SOCK_STREAM, 0);

  memset(&sin, 0, sizeof(sin));
  sin.sin_port = htons(port);
  
  if (bind(sock, (struct sockaddr *) &sin, sizeof (sin)) < 0) {
    fprintf(stderr,
            "Can't bind to *:%d.  Check for conflicting servers.\n",
            port);
    return -1;
  }

  val = 1;
  if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &val, sizeof(val)) < 0) {
    fprintf(stderr, "internal error: bad set sock opt\n");
    return -1;
  }
  
  listen(sock, 5);

  return sock;
}

void
set_classpath(JavaVMOption *options)
{
  char classpath[8192];
  char def[8192];
  char *resin_home = "/home/ferg/ws/resin";
  char path[1024];
  DIR *dir;
  struct dirent *dir_entry;

  sprintf(path, "%s/lib", resin_home);

  sprintf(classpath, "%s/classes", resin_home);
  
  dir = opendir(path);
  while ((dir_entry = readdir(dir))) {
    int len = strlen(dir_entry->d_name);
    
    if (! strcmp(dir_entry->d_name + len - 4, ".jar") ||
        ! strcmp(dir_entry->d_name + len - 4, ".zip")) {
      strcat(classpath, ":");
      strcat(classpath, resin_home);
      strcat(classpath, "/lib/");
      strcat(classpath, dir_entry->d_name);
    }
  }
  closedir(dir);
  
  if (getenv("CLASSPATH")) {
    strcat(classpath, ":");
    strcat(classpath, getenv("CLASSPATH"));
  }

  sprintf(def, "-Djava.class.path=%s", classpath);
  
  options[0].optionString = def;
  
  //AddOption(def, NULL);
}

/**
 * Returns true if prefix + tail is a file.
 */
static int
is_file(char *prefix, char *tail)
{
  struct stat st;
  char buf[8192];

  sprintf(buf, "%s/%s", prefix, tail);

  return ! stat(buf, &st) && S_ISREG(st.st_mode);
}

void
set_library_path(char *java_home, char *resin_home)
{
  char envpath[8192];
  char newpath[8192];

  char *type;

  strcpy(newpath, resin_home);
  strcat(newpath, "/libexec:");
  
  if (is_file(java_home, "jre/bin/libjava.so")) {
    type = "classic";
    
    strcat(newpath, java_home);
    strcat(newpath, "/jre/bin:");

    strcat(newpath, java_home);
    strcat(newpath, "/jre/bin/");
    strcat(newpath, type);
    strcat(newpath, ":");
  }
  else if (is_file(java_home, "jre/lib/" OS "/libjava.so")) {
    type = "server";
 
    strcat(newpath, java_home);
    strcat(newpath, "/jre/lib/" OS ":");

    strcat(newpath, java_home);
    strcat(newpath, "/jre/lib/" OS "/");
    strcat(newpath, type);
    strcat(newpath, ":");
    
    strcat(newpath, java_home);
    strcat(newpath, "/jre/lib/" OS "/native_threads");
    strcat(newpath, ":");
  }
  
  strcat(newpath, getenv("LD_LIBRARY_PATH"));

  strcpy(envpath, "LD_LIBRARY_PATH=");
  strcat(envpath, newpath);
  putenv(strdup(envpath));
}

void
setEnvVars(char *java_home)
{
  char newpath[8192];
  char *p;
  char *buffer = newpath;

    /* Now add our locale files to the beginning of XFILESEARCHPATH */

  strcpy(buffer, "XFILESEARCHPATH=");
  strcat(buffer, java_home);
  strcat(buffer, "/jre/lib/locale/%L/%T/%N%S:");

  p = getenv("XFILESEARCHPATH");
  if (p != NULL) {
    strcat(buffer, p);
  }
    
  putenv(strdup(buffer));
}

int cse_test() { return 666; }

static int
create_vm(JavaVM **p_vm, void **env, JavaVMInitArgs *vm_args)
{
  void *handle;
  
  int (*p_create_vm) (JavaVM **, void **, JavaVMInitArgs *);
  
  handle = dlopen("libjvm.so", RTLD_LAZY);
  if (! handle) {
    fprintf(stderr, "%s\n", dlerror());
    exit(1);
  }

  p_create_vm = dlsym(handle, "JNI_CreateJavaVM");
  if (! p_create_vm) {
    fprintf(stderr, "can't find JNI_CreateJavaVM\n");
    exit(1);
  }

  return p_create_vm(p_vm, env, vm_args);
}

/**
 * Parses the command-line arguments
 */
static void
parse_command_line(int argc, char **argv)
{
  int i;
  
  for (i = 1; i < argc; i++) {
    if (! strcmp(argv[i], "-java_home") ||
        ! strcmp(argv[i], "-java-home")) {
      g_options.java_home = strdup(argv[i + 1]);
      i++;
    }
    else if (! strcmp(argv[i], "-resin_home") ||
             ! strcmp(argv[i], "-resin-home")) {
      g_options.resin_home = strdup(argv[i + 1]);
      i++;
    }
    else if (! strcmp(argv[i], "-conf") ||
             ! strcmp(argv[i], "-conf")) {
      g_options.conf = strdup(argv[i + 1]);
      i++;
    }
    else if (! strcmp(argv[i], "-classpath") ||
             ! strcmp(argv[i], "-cp")) {
      g_options.classpath = strdup(argv[i + 1]);
      i++;
    }
    else if (! strcmp(argv[i], "-loadpath") ||
             ! strcmp(argv[i], "-lp")) {
      g_options.loadpath = strdup(argv[i + 1]);
      i++;
    }
  }
}

/**
 * unwinds a linked path, returning the original directory.
 */
static char *
unwind_link(char *path, char *test_tail)
{
  struct stat st;
  int tail = 0;

  while ((! lstat(path, &st)) && S_ISLNK(st.st_mode)) {
    char buf[8192];

    readlink(path, buf, sizeof(buf));

    path = strdup(buf);
  }

  for (tail = strlen(path); tail >= 0; tail--) {
    if (tail == '/') {
      path[tail] = 0;

      if (is_file(path, test_tail))
        return path;
    }
  }

  return 0;
}

/**
 * Finds a link in a classpath-like path
 */
static char *
find_link_in_path(char *path, char *test_tail, char *exe)
{
  char *head;
  char *tail = 0;

  for (head = path; head && *head; head = tail) {
    char buf[8192];
    struct stat st;
      
    tail = strchr(head, ':');

    if (tail) {
      *tail = 0;
      tail++;
    }

    sprintf(buf, "%s/%s", head, exe);
      
    if (! stat(buf, &st)) {
      char *dir = unwind_link(buf, test_tail);

      if (dir)
        return dir;
    }
  }

  return 0;
}

/**
 * Finds the value of java_home
 *
 * 1) command-line
 * 2) getenv("JAVA_HOME")
 * 3) find by searching back the "java" command
 * 4) trying /usr/java
 */
static void
find_java_home()
{
  char *java_home;
  char *path;
  
  if (g_options.java_home)
    return;

  if ((java_home = getenv("JAVA_HOME"))) {
    g_options.java_home = strdup(java_home);
    return;
  }

  path = getenv("PATH");
  if (path) {
    path = strdup(path);
    
    g_options.java_home = find_link_in_path(path, "jre/lib/rt.jar", "java");
  }

  if (! g_options.java_home)
    g_options.java_home = "/usr/java";
}

/**
 * Finds the value of resin_home
 *
 * 1) command-line
 * 2) getenv("RESIN_HOME")
 * 3) trying /usr/local/resin
 */
static void
find_resin_home(char *cmd)
{
  char *resin_home;
  char *path;
  
  if (g_options.resin_home)
    return;

  if ((resin_home = getenv("RESIN_HOME"))) {
    g_options.resin_home = strdup(resin_home);
    return;
  }

  g_options.resin_home = unwind_link(cmd, "lib/jsdk22.jar");
  if (g_options.resin_home)
    return;

  if (! g_options.resin_home)
    g_options.resin_home = "/home/ferg/ws/resin";
}

static void
init_java(JNIEnv *env)
{
  jclass classID;
  jmethodID methodID;
  jfieldID fieldID;
  
  jni_vfs_init(env);

  classID = (*env)->FindClass(env, "com/caucho/server/http/JniServer");
  if (classID == 0) {
    fprintf(stderr, "Class not found: com.caucho.server.http.JniServer\n");
    exit(1);
  }
  g_java.jni_server = classID;

  classID = (*env)->FindClass(env, "com/caucho/server/http/JniRequest");
  if (classID == 0) {
    fprintf(stderr, "Class not found: com.caucho.server.http.JniRequest\n");
    exit(1);
  }
  g_java.jni_request = classID;

  /* Get the application's main method */
  methodID = (*env)->GetMethodID(env, g_java.jni_server, "<init>", "()V");
  if (! methodID) {
    fprintf(stderr, "Can't instantiate com.caucho.server.http.JniServer\n");
    exit(1);
  }
  g_java.new_jni_server = methodID;
  
  methodID = (*env)->GetMethodID(env, g_java.jni_server, "createRequest",
                                 "()Lcom/caucho/server/http/JniRequest;");
  if (! methodID) {
    fprintf(stderr, "Can't find createRequest");
    exit(1);
  }
  g_java.create_request = methodID;
  
  methodID = (*env)->GetMethodID(env, g_java.jni_request,
                                 "dispatch", "(IIIII)V");
  g_java.dispatch = methodID;

  fieldID = (*env)->GetFieldID(env, g_java.jni_request, "uri", "[B");
  g_java.f_uri = fieldID;
  
  fieldID = (*env)->GetFieldID(env, g_java.jni_request, "uriLength", "I");
  g_java.f_uri_length = fieldID;
  
  fieldID = (*env)->GetFieldID(env, g_java.jni_request, "headerBuffer", "[C");
  g_java.f_header_buffer = fieldID;
  
  fieldID = (*env)->GetFieldID(env, g_java.jni_request, "headerOffsets", "[I");
  g_java.f_header_offsets = fieldID;
  
  fieldID = (*env)->GetFieldID(env, g_java.jni_request, "headerSize", "I");
  g_java.f_header_size = fieldID;
}

int
main(int argc, char **argv)
{
  int ss;

  JavaVM *vm = 0;
  JNIEnv *env = 0;
  JavaVMInitArgs vm_args;
  JavaVMOption options[4];
  jobject server_obj = 0;
  jobject request_obj = 0;
  char envpath[8192];
  int res;
  request_t *request = request_create();
  response_t *response = response_create();

  parse_command_line(argc, argv);
  
  memset(&vm_args, 0, sizeof(vm_args));
  memset(options, 0, sizeof(options));

  find_java_home();
  find_resin_home(argv[0]);

  if (argc < 2 || strcmp(argv[1], "-XXX:start")) {
    char **newargv = malloc((argc + 1) * sizeof(char *));
    newargv[0] = argv[0];
    newargv[1] = "-XXX:start";
    memcpy(newargv + 2, argv + 1, argc * sizeof(char *));

    set_library_path(g_options.java_home, g_options.resin_home);

    execv(argv[0], newargv);

    fprintf(stderr, "can't start %s\n", argv[0]);
    exit(1);
  }
  
  ss = cse_bind("*", 8080);
  if (ss < 0)
    exit(1);

  // XXX: now, change ownership

  setEnvVars(g_options.java_home);
  set_classpath(options);
  
  strcpy(envpath, "-Djava.library.path=");
  strcat(envpath, getenv("LD_LIBRARY_PATH"));

  options[1].optionString = strdup(envpath);

  options[2].optionString = "-Dresin.home=/home/ferg/ws/resin";
  //options[3].optionString = "-verbose:jni";

  vm_args.version = JNI_VERSION_1_2;
  vm_args.options = options;
  vm_args.nOptions = 3;
  vm_args.ignoreUnrecognized = 0;

  res = create_vm(&vm, (void **)&env, &vm_args);

  g_java.vm = vm;

  init_java(env);

  server_obj = (*env)->NewObject(env, g_java.jni_server,
                                 g_java.new_jni_server);
  if ((*env)->ExceptionOccurred(env)) {
    (*env)->ExceptionDescribe(env);
    exit(1);
  }
  server_obj = (*env)->NewGlobalRef(env, server_obj);

  start_server(ss, env, server_obj);
}
