/*
 * Copyright (c) 1999-2000 Caucho Technology.  All rights reserved.
 *
 * Caucho Technology permits redistribution, modification and use
 * of this file in source and binary form ("the Software") under the
 * Caucho Developer Source License ("the License").  In particular, the following
 * conditions must be met:
 *
 * 1. Each copy or derived work of the Software must preserve the copyright
 *    notice and this notice unmodified.
 *
 * 2. Redistributions of the Software in source or binary form must include 
 *    an unmodified copy of the License, normally in a plain ASCII text
 *
 * 3. The names "Resin" or "Caucho" are trademarks of Caucho Technology and
 *    may not be used to endorse products derived from this software.
 *    "Resin" or "Caucho" may not appear in the names of products derived
 *    from this software.
 *
 * 4. Caucho Technology requests that attribution be given to Resin
 *    in any manner possible.  We suggest using the "Resin Powered"
 *    button or creating a "powered by Resin(tm)" link to
 *    http://www.caucho.com for each page served by Resin.
 *
 * This Software is provided "AS IS," without a warranty of any kind. 
 * ALL EXPRESS OR IMPLIED REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.

 * Caucho Technology AND ITS LICENSORS SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE OR ANY THIRD PARTY AS A RESULT OF USING OR
 * DISTRIBUTING SOFTWARE. IN NO EVENT WILL Caucho OR ITS LICENSORS BE LIABLE
 * FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL,
 * CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND
 * REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
 * INABILITY TO USE SOFTWARE, EVEN IF HE HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGES.      
 */

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <time.h>
#include <stdlib.h>
#ifdef WIN32
#include <windows.h>
#endif
#include "cse.h"

#define BLOCK_SIZE 32 * 1024

typedef struct pool_block_t {
  struct pool_block_t *next;

  char *buffer;
  int size;
  int offset;
} pool_block_t;

struct mem_pool_t {
  struct pool_block_t *block;
  void *lock;
};

mem_pool_t *
cse_create_pool(config_t *config)
{
  mem_pool_t *pool = (mem_pool_t *) cse_malloc(sizeof(mem_pool_t));
  memset(pool, 0, sizeof(pool));
  pool->lock = cse_create_lock(config);

  return pool;
}

void *
cse_alloc(mem_pool_t *pool, int size)
{
  void *data;
  int frag = size % 8;

  if (frag > 0)
    size += 8 - frag;

  cse_lock(pool->lock);

  if (! pool->block || pool->block->size - pool->block->offset < size) {
    pool_block_t *block = (pool_block_t *) cse_malloc(sizeof(pool_block_t));
    memset(block, 0, sizeof(pool_block_t));
    block->next = pool->block;
    pool->block = block;

    if (size > BLOCK_SIZE)
      block->size = size;
    else
      block->size = BLOCK_SIZE;

    block->buffer = cse_malloc(block->size);
  }

  data = pool->block->buffer + pool->block->offset;
  pool->block->offset += size;
  
  cse_unlock(pool->lock);

  return data;
}

char *
cse_strdup(mem_pool_t *pool, const char *str)
{
  int len = strlen(str);
  char *buf = cse_alloc(pool, len + 1);

  strcpy(buf, str);

  return buf;
}

void
cse_free_pool(mem_pool_t *pool)
{
  pool_block_t *ptr;
  pool_block_t *next;

  cse_lock(pool->lock);

  for (ptr = pool->block; ptr; ptr = next) {
    next = ptr->next;

    cse_free(ptr->buffer);
    cse_free(ptr);
  }

  pool->block = 0;
  
  cse_unlock(pool->lock);
}
