/*
 * Copyright (c) 1999-2000 Caucho Technology.  All rights reserved.
 *
 * Caucho Technology permits redistribution, modification and use
 * of this file in source and binary form ("the Software") under the
 * Caucho Developer Source License ("the License").  In particular, the following
 * conditions must be met:
 *
 * 1. Each copy or derived work of the Software must preserve the copyright
 *    notice and this notice unmodified.
 *
 * 2. Redistributions of the Software in source or binary form must include 
 *    an unmodified copy of the License, normally in a plain ASCII text
 *
 * 3. The names "Resin" or "Caucho" are trademarks of Caucho Technology and
 *    may not be used to endorse products derived from this software.
 *    "Resin" or "Caucho" may not appear in the names of products derived
 *    from this software.
 *
 * 4. Caucho Technology requests that attribution be given to Resin
 *    in any manner possible.  We suggest using the "Resin Powered"
 *    button or creating a "powered by Resin(tm)" link to
 *    http://www.caucho.com for each page served by Resin.
 *
 * This Software is provided "AS IS," without a warranty of any kind. 
 * ALL EXPRESS OR IMPLIED REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.

 * CAUCHO TECHNOLOGY AND ITS LICENSORS SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE OR ANY THIRD PARTY AS A RESULT OF USING OR
 * DISTRIBUTING SOFTWARE. IN NO EVENT WILL Caucho OR ITS LICENSORS BE LIABLE
 * FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL,
 * CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND
 * REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
 * INABILITY TO USE SOFTWARE, EVEN IF HE HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGES.      
 */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/times.h>
#include <sys/resource.h>
#include <pwd.h>
#include <grp.h>
#include <unistd.h>

#include <jni.h>

static struct rusage rusage_self;
static struct rusage rusage_children;
static struct tms cpu_times;

void
Java_com_caucho_util_CauchoNative_calculateUsage()
{
  getrusage(RUSAGE_SELF, &rusage_self);
  getrusage(RUSAGE_CHILDREN, &rusage_children);
  times(&cpu_times);
}

static double
time_as_double(struct timeval time)
{
  return time.tv_sec + time.tv_usec * 1e-6;
}

int
Java_com_caucho_util_CauchoNative_getPid()
{
  return getpid();
}

double
Java_com_caucho_util_CauchoNative_getUserTime()
{
  return time_as_double(rusage_self.ru_utime);
}

double
Java_com_caucho_util_CauchoNative_getSystemTime()
{
  return time_as_double(rusage_self.ru_stime);
}

long
Java_com_caucho_util_CauchoNative_getMaxResidentSetSize()
{
  return rusage_self.ru_maxrss;
}

long
Java_com_caucho_util_CauchoNative_getResidentSetSize()
{
  return (rusage_self.ru_ixrss +
	  rusage_self.ru_idrss +
	  rusage_self.ru_isrss);
}

long
Java_com_caucho_util_CauchoNative_getSwaps()
{
  return rusage_self.ru_nswap;
}

long
Java_com_caucho_util_CauchoNative_getContextSwitches()
{
  return rusage_self.ru_nvcsw + rusage_self.ru_nivcsw;
}

jboolean
Java_com_caucho_util_CauchoNative_setUser(JNIEnv *env, jobject obj,
                                          jstring user, jstring groupName)
{
  struct passwd *passwd;
  struct group *group;
  
  jboolean result = 0;
  
  int uid = -1;
  int gid = -1;
  
  const char *name;

  if (! user)
    return 0;

  name = (*env)->GetStringUTFChars(env, user, 0);
  passwd = getpwnam(name);
  (*env)->ReleaseStringUTFChars(env, user, name);
  
  if (passwd) {
    uid = passwd->pw_uid;
    gid = passwd->pw_gid;
  }

  if (groupName) {
    name = (*env)->GetStringUTFChars(env, groupName, 0);
    group = getgrnam(name);
    (*env)->ReleaseStringUTFChars(env, groupName, name);
    if (group) {
      gid = group->gr_gid;
    }
  }

  if (uid >= 0) {
    result = 1;

    /*
    if (gid >= 0)
      result = setgid(gid) >= 0;
    */

    if (result > 0) {
      result = setuid(uid) >= 0;
    }
  }

  result = 1;

  return result;
}
