/*
 * Copyright (c) 1998-2002 Caucho Technology -- all rights reserved
 *
 * Caucho Technology permits modification and use of this file in
 * source and binary form ("the Software") subject to the Caucho
 * Developer Source License 1.1 ("the License") which accompanies
 * this file.  The License is also available at
 *   http://www.caucho.com/download/cdsl1-1.xtp
 *
 * In addition to the terms of the License, the following conditions
 * must be met:
 *
 * 1. Each copy or derived work of the Software must preserve the copyright
 *    notice and this notice unmodified.
 *
 * 2. Each copy of the Software in source or binary form must include 
 *    an unmodified copy of the License in a plain ASCII text file named
 *    LICENSE.
 *
 * 3. Caucho reserves all rights to its names, trademarks and logos.
 *    In particular, the names "Resin" and "Caucho" are trademarks of
 *    Caucho and may not be used to endorse products derived from
 *    this software.  "Resin" and "Caucho" may not appear in the names
 *    of products derived from this software.
 *
 * This Software is provided "AS IS," without a warranty of any kind. 
 * ALL EXPRESS OR IMPLIED REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 *
 * CAUCHO TECHNOLOGY AND ITS LICENSORS SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE OR ANY THIRD PARTY AS A RESULT OF USING OR
 * DISTRIBUTING SOFTWARE. IN NO EVENT WILL CAUCHO OR ITS LICENSORS BE LIABLE
 * FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL,
 * CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND
 * REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
 * INABILITY TO USE SOFTWARE, EVEN IF HE HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGES.      
 *
 * @author Scott Ferguson
 */


#ifdef WIN32
#ifndef _WINSOCKAPI_ 
#define _WINSOCKAPI_
#endif 
#include <windows.h>
#include <winsock2.h> 
#else
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <unistd.h>
#include <sys/time.h>
#include <pwd.h>
#endif

#ifdef linux
#include <linux/version.h>
#endif

#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
/* probably system-dependent */
#include <jni.h>

#include "../common/cse.h"
#include "resin.h"
#include "resin_jni.h"

void
cse_log(char *fmt, ...)
{
#ifdef DEBUG  
  va_list list;

  va_start(list, fmt);
  vfprintf(stderr, fmt, list);
  va_end(list);
#endif
}

int
Java_com_caucho_vfs_JniStream_readNative(JNIEnv *env,
                                         jobject obj,
                                         jint fd,
                                         jbyteArray buf,
                                         jint offset,
                                         jint length)
{
  connection_t *conn = (connection_t *) fd;
  int sublen;
  char *jbuf;
  jboolean is_copy = 0;
  char buffer[8192];

  if (! conn || conn->fd < 0)
    return -1;

  if (length < 8192)
    sublen = length;
  else
    sublen = 8192;

  sublen = conn->ops->read(conn, buffer, sublen);

  /* Should probably have a different response for EINTR */
  if (sublen < 0)
    return sublen;

  jbuf = (*env)->GetPrimitiveArrayCritical(env, buf, &is_copy);

  if (jbuf)
    memcpy(jbuf + offset, buffer, sublen);
  
  (*env)->ReleasePrimitiveArrayCritical(env, buf, jbuf, 0);

  return sublen;
}

int
Java_com_caucho_vfs_JniStream_writeNative(JNIEnv *env,
                                          jobject obj,
                                          jint fd,
                                          jbyteArray buf,
                                          jint offset,
                                          jint length)
{
  connection_t *conn = (connection_t *) fd;
  char *cbuf;
  char buffer[8192];
  jboolean is_copy;
  int sublen;
  int buffer_offset;

  buffer_offset = 0;
  
  while (conn && length > 0 && conn) {
    if (length < 8192)
      sublen = length;
    else
      sublen = 8192;

    is_copy = 0;
    cbuf = (*env)->GetPrimitiveArrayCritical(env, buf, &is_copy);

    if (cbuf)
      memcpy(buffer, cbuf + offset, sublen);

    (*env)->ReleasePrimitiveArrayCritical(env, buf, cbuf, 0);

    if (cbuf) {
      sublen = conn->ops->write(conn, buffer, sublen);
    }
    else
      return -1;

    if (sublen < 0)
      return sublen;

    length -= sublen;
    offset += sublen;
  }

  return 0;
}

int
Java_com_caucho_vfs_JniStream_flushNative(JNIEnv *env,
                                          jobject obj,
                                          jint fd)
{
  connection_t *conn = (connection_t *) fd;

  if (! conn)
    return -1;
  else
    return 0;

  /* return cse_flush_request(res); */
}

void
Java_com_caucho_vfs_JniStream_closeNative(JNIEnv *env,
                                          jobject obj,
                                          jint fd)
{
  connection_t *conn = (connection_t *) fd;

  if (conn) {
    conn->ops->close(conn);
  }
}

void
Java_com_caucho_vfs_QJniSocket_closeNative(JNIEnv *env,
                                          jobject obj,
                                          jint fd)
{
  connection_t *conn = (connection_t *) fd;

  if (conn) {
    conn->ops->close(conn);
    conn->ops->free(conn);
  }
}

jboolean
Java_com_caucho_vfs_QJniSocket_isSecure(JNIEnv *env,
                                        jobject obj,
                                        jint fd)
{
  connection_t *conn = (connection_t *) fd;

  if (! conn)
    return 0;
  
  return conn->ssl_cipher != 0;
}

jstring
Java_com_caucho_vfs_QJniSocket_getCipher(JNIEnv *env,
                                         jobject obj,
                                         jint fd)
{
  connection_t *conn = (connection_t *) fd;

  if (! conn || ! conn->ssl_cipher)
    return 0;

  return (*env)->NewStringUTF(env, conn->ssl_cipher);
}

jint
Java_com_caucho_vfs_QJniSocket_getCipherBits(JNIEnv *env,
                                         jobject obj,
                                         jint fd)
{
  connection_t *conn = (connection_t *) fd;

  if (! conn)
    return 0;
  else
    return conn->ssl_bits;
}

void
Java_com_caucho_vfs_QJniSocket_setTimeout(JNIEnv *env,
                                          jobject obj,
                                          jint fd,
                                          jint timeout)
{
  connection_t *conn = (connection_t *) fd;

  if (! conn)
    return;

  conn->timeout = timeout;
}

jint
Java_com_caucho_vfs_QJniServerSocket_acceptNative(JNIEnv *env,
                                                  jobject obj,
                                                  jint ss)
{
  server_socket_t *socket = (server_socket_t *) ss;
  connection_t *conn;

  if (! socket)
    return 0;

  conn = socket->accept(socket);

  return (int) conn;
}

jint
Java_com_caucho_vfs_QJniServerSocket_getLocalPort(JNIEnv *env,
                                                  jobject obj,
                                                  jint ss)
{
  server_socket_t *socket = (server_socket_t *) ss;

  if (! socket)
    return 0;
  else
    return socket->port;
}

jint
Java_com_caucho_server_http_JniServer_getServerSocket(JNIEnv *env,
                                                      jobject obj,
                                                      jint resin_fd)
{
  resin_t *resin = (resin_t *) resin_fd;
  int ss;

  if (! resin)
    return -1;

  ss = resin->get_server_socket(resin);
  
  return ss;
}

jlong
Java_com_caucho_vfs_QJniSocket_getRemoteIP(JNIEnv *env,
                                           jobject obj,
                                           jint fd)
{
  connection_t *conn = (connection_t *) fd;

  if (! conn)
    return 0;

  return ntohl(conn->client_sin.sin_addr.s_addr);
}

jint
Java_com_caucho_vfs_QJniSocket_getRemotePort(JNIEnv *env,
                                             jobject obj,
                                             jint fd)
{
  connection_t *conn = (connection_t *) fd;

  if (! conn)
    return 0;

  return ntohs(conn->client_sin.sin_port);
}

jlong
Java_com_caucho_vfs_QJniSocket_getLocalIP(JNIEnv *env,
                                           jobject obj,
                                           jint fd)
{
  connection_t *conn = (connection_t *) fd;

  if (! conn)
    return 0;

  return ntohl(conn->server_sin.sin_addr.s_addr);
}

jint
Java_com_caucho_vfs_QJniSocket_getLocalPort(JNIEnv *env,
                                             jobject obj,
                                             jint fd)
{
  connection_t *conn = (connection_t *) fd;

  if (! conn)
    return 0;

  return ntohs(conn->server_sin.sin_port);
}

jboolean
Java_com_caucho_server_http_JniServer_nativeWaitForExit(JNIEnv *env,
                                                        jobject obj)
{
  fd_set read_mask;
  struct timeval timeout;
  char buf[4];
  
  FD_ZERO(&read_mask);
  FD_SET(0, &read_mask);
  
  timeout.tv_sec = 10;
  timeout.tv_usec = 0;
  
  if (select(1, &read_mask, 0, 0, &timeout) <= 0)
    return 0;
  else
    return read(0, buf, 1) <= 0;
}

jint
Java_com_caucho_vfs_QJniSocket_getClientCertificate(JNIEnv *env,
                                                    jobject obj,
                                                    jint fd,
                                                    jbyteArray buf,
                                                    jint offset,
                                                    jint length)
{
  connection_t *conn = (connection_t *) fd;
  int sublen;
  char *jbuf;
  jboolean is_copy = 0;
  char buffer[8192];

  if (! conn)
    return -1;

  if (length < 8192)
    sublen = length;
  else
    sublen = 8192;

  sublen = conn->ops->read_client_certificate(conn, buffer, sublen);

  /* Should probably have a different response for EINTR */
  if (sublen < 0 || sublen > length)
    return sublen;

  jbuf = (*env)->GetPrimitiveArrayCritical(env, buf, &is_copy);

  if (jbuf)
    memcpy(jbuf + offset, buffer, sublen);
  
  (*env)->ReleasePrimitiveArrayCritical(env, buf, jbuf, 0);

  return sublen;
}

static char *
q_strdup(char *str)
{
  int len = strlen(str);
  char *dup = cse_malloc(len + 1);

  strcpy(dup, str);

  return dup;
}

static int
get_utf(JNIEnv *env, jstring jString, char *buffer)
{
  const char *temp_string;
  
  temp_string = (*env)->GetStringUTFChars(env, jString, 0);
  
  if (temp_string)
    strcpy(buffer, temp_string);
  
  (*env)->ReleaseStringUTFChars(env, jString, temp_string);

  return temp_string != 0;
}

static char *
strdup_utf(JNIEnv *env, jstring jString)
{
  const char *temp_string;
  char buffer[1024];

  if (jString == 0)
    return 0;
  
  temp_string = (*env)->GetStringUTFChars(env, jString, 0);
  
  if (temp_string)
    strcpy(buffer, temp_string);
  
  (*env)->ReleaseStringUTFChars(env, jString, temp_string);

  return temp_string != 0 ? strdup(buffer) : 0;
}

int
Java_com_caucho_vfs_QSSLFactory_bindPort(JNIEnv *env,
                                         jobject obj,
                                         jlong addr,
                                         jint port,
                                         jstring jCertificateFile,
                                         jstring jKeyFile,
                                         jstring jCertificateChainFile,
                                         jstring jCaCertificatePath,
                                         jstring jCaCertificateFile,
                                         jstring jCaRevocationPath,
                                         jstring jCaRevocationFile,
                                         jstring jPassword,
                                         jstring jVerifyClient)
{
  struct sockaddr_in sin;
  int val = 0;
  int sock;
  server_socket_t *ss;
  ssl_config_t *config;
  const char *string;
  char certificate_file[1024];
  char key_file[1024];
  char password[1024];

  if (! jCertificateFile)
    return 0;

  if (! jKeyFile)
    return 0;

  if (! jPassword)
    return 0;

  if (! get_utf(env, jCertificateFile, certificate_file))
    return 0;
  
  if (! get_utf(env, jKeyFile, key_file))
    return 0;
  
  if (! get_utf(env, jPassword, password))
    return 0;

  memset(&sin, 0, sizeof(sin));

  sin.sin_port = htons((unsigned short) port);

  sock = socket(AF_INET, SOCK_STREAM, 0);
  
  val = 1;
  if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR,
                 (char *) &val, sizeof(int)) < 0) {
    closesocket(sock);
    return 0;
  }
  
  if (bind(sock, (struct sockaddr *) &sin, sizeof (sin)) < 0) {
    closesocket(sock);
    return 0;
  }

  listen(sock, 5);

  ss = (server_socket_t *) cse_malloc(sizeof(server_socket_t));
  memset(ss, 0, sizeof(server_socket_t));

  ss->fd = sock;
  ss->port = port;
  
  pthread_mutex_init(&ss->ssl_lock, 0);
  
  ss->request_timeout = 30;

  if (ss->request_timeout <= 0)
    ss->request_timeout = 30;
    
  if (ss->request_timeout < 10)
    ss->request_timeout = 10;

  config = cse_malloc(sizeof(ssl_config_t));
  memset(config, 0, sizeof(ssl_config_t));

  config->certificate_file = q_strdup(certificate_file);
  config->key_file = q_strdup(key_file);
  config->password = q_strdup(password);
  config->alg_flags = ~0;
  if (! get_utf(env, jPassword, password))
    return 0;
  
  config->certificate_chain_file = strdup_utf(env, jCertificateChainFile);
  config->ca_certificate_path = strdup_utf(env, jCaCertificatePath);
  config->ca_certificate_file = strdup_utf(env, jCaCertificateFile);
  config->ca_revocation_path = strdup_utf(env, jCaRevocationPath);
  config->ca_revocation_file = strdup_utf(env, jCaRevocationFile);

  if (jVerifyClient) {
    string = (*env)->GetStringUTFChars(env, jVerifyClient, 0);

    if (! string) {
    }
    else if (! strcmp(string, "optional_no_ca"))
      config->verify_client = Q_VERIFY_OPTIONAL_NO_CA;
    else if (! strcmp(string, "optional"))
      config->verify_client = Q_VERIFY_OPTIONAL;
    else if (! strcmp(string, "require"))
      config->verify_client = Q_VERIFY_REQUIRE;
  
    (*env)->ReleaseStringUTFChars(env, jVerifyClient, string);
    
    if (! string)
      return 0;
  }

  /* memory leak with the config. */
  if (! ssl_create(ss, config)) {
    cse_free(ss);
    closesocket(sock);
    return 0;
  }
  
  return (int) ss;
}

#ifdef linux
static void
get_linux_version(char *version)
{
  FILE *file = fopen("/proc/version", "r");

  if (! file || fscanf(file, "Linux version %s", version) != 1)
    strcpy(version, "2.4.0-unknown");

  fclose(file);
}
#endif

#ifndef WIN32
jint
Java_com_caucho_util_CauchoSystem_setUserNative(JNIEnv *env,
						jobject obj,
						jstring juser,
						jstring jgroup)
{
  char userbuf[256];
  char groupbuf[256];
  char *user = 0;
  char *group = 0;
  int uid = -1;
  int gid = -1;
  const char *temp_string;
  struct passwd *passwd;
  
  if (juser != 0) {
    temp_string = (*env)->GetStringUTFChars(env, juser, 0);
  
    if (temp_string) {
      strncpy(userbuf, temp_string, sizeof(userbuf));
      userbuf[sizeof(userbuf) - 1] = 0;
      user = userbuf;
    }
  
    (*env)->ReleaseStringUTFChars(env, juser, temp_string);
  }

  if (jgroup != 0) {
    temp_string = (*env)->GetStringUTFChars(env, jgroup, 0);
  
    if (temp_string) {
      strncpy(groupbuf, temp_string, sizeof(groupbuf));
      groupbuf[sizeof(groupbuf) - 1] = 0;
      group = groupbuf;
    }
  
    (*env)->ReleaseStringUTFChars(env, jgroup, temp_string);
  }

  if (user == 0)
    return -1;

  passwd = getpwnam(user);
  if (passwd == 0)
    return -1;

  uid = passwd->pw_uid;

  if (group) {
    passwd = getpwnam(group);
    if (passwd)
      gid = passwd->pw_gid;
  }

  if (uid == getuid())
    return uid;

  /* XXX: should be a better check for the new thread library. */
#ifdef linux_check
  {
    char buf[1024];
    char version[1024];
    jclass clazz;

    get_linux_version(version);

    if (strcmp(version, "2.5.0") < 0) {
      sprintf(buf, "Linux %s can't use setuid.\nThe <user-name> tag requires Linux kernel 2.5 or later.\nSee the documentation for alternative ways of handling protected ports.",
	      version);

      clazz = (*env)->FindClass(env, "java/io/IOException");

      if (clazz)
	(*env)->ThrowNew(env, clazz, buf);

      return -1;
    }
  }
#endif

  if (gid > 0)
    setgid(gid);

  if (uid > 0) {
    int result;

    result = setuid(uid);
    
    if (result > 0)
      return -1;
  }

  return getuid();
}
#else /* WIN32 */
jint
Java_com_caucho_util_CauchoSystem_setUserNative(JNIEnv *env,
						jobject obj,
						jstring user,
						jstring group)
{
  return -1;
}
#endif
