/*
 * Copyright (c) 1998-2001 Caucho Technology -- all rights reserved
 *
 * Caucho Technology permits redistribution, modification and use
 * of this file in source and binary form ("the Software") under the
 * Caucho Developer Source License ("the License").  The following
 * conditions must be met:
 *
 * 1. Each copy or derived work of the Software must preserve the copyright
 *    notice and this notice unmodified.
 *
 * 2. Redistributions of the Software in source or binary form must include 
 *    an unmodified copy of the License, normally in a plain ASCII text
 *
 * 3. The names "Resin" or "Caucho" are trademarks of Caucho Technology and
 *    may not be used to endorse products derived from this software.
 *    "Resin" or "Caucho" may not appear in the names of products derived
 *    from this software.
 *
 * This Software is provided "AS IS," without a warranty of any kind. 
 * ALL EXPRESS OR IMPLIED REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 *
 * CAUCHO TECHNOLOGY AND ITS LICENSORS SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE OR ANY THIRD PARTY AS A RESULT OF USING OR
 * DISTRIBUTING SOFTWARE. IN NO EVENT WILL CAUCHO OR ITS LICENSORS BE LIABLE
 * FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL,
 * CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND
 * REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
 * INABILITY TO USE SOFTWARE, EVEN IF HE HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGES.      
 */

package example.servlet.basic;

import java.io.*;
import javax.servlet.*;
import javax.servlet.http.*;

/**
 * Shows how to extract the URL components.
 *
 * <pre>
 * http://localhost:8080/tutorial/servlet/example.URLServlet/foo?a=b
 * </pre>
 *
 * <table>
 * <tr><td><b>getScheme</b><td>http
 * <tr><td><b>getServerName</b><td>localhost
 * <tr><td><b>getServerPort</b><td>8080
 * <tr><td><b>getContextPath</b><td>/tutorial
 * <tr><td><b>getServletPath</b><td>/servlet/example.URLServlet
 * <tr><td><b>getPathInfo</b><td>/foo
 * <tr><td><b>getQueryString</b><td>a=b
 * </table>
 *
 * <p>The contextPath, servletPath, and pathInfo are related to how
 * URLs map to servlets.
 *
 * <p>Each host is composed of one or more web-applications, specified
 * by a URI prefix, the contextPath.  This tutorial, for example,
 * is in a web-app named /tutorial.   A web-app is a
 * miniature virtual host.  The web-apps are protected from each other
 * and the classes are entirely distinct.  Because web-apps are
 * often renamed, the servlets should be written not to care what the
 * contextPath is.  For example, any absolute URL needs to add the
 * contextPath.  If the servlet wanted a link to /tutorial/foo.jsp,
 * it would use:
 *
 * <code><pre>
 * out.println("&lt;a href=\"" + request.getContextPath() + "/foo.jsp>");
 * </pre></code>
 *
 * <p>servletPath is the URL that maps directly to the servlet.
 * Examples would be /servlet/example.URLServlet or /test/foo.jsp.
 * (Since *.jsp files are really servlets.)
 *
 * <p>pathInfo is the part of the URL after the servlet path.  Servlets
 * will often use the pathInfo as some sort of database key.
 *
 * <p>queryString is the query part of the URL.  Normally, servlets
 * use the getParameter call instead of parting the query string directly.
 */
public class URLServlet extends HttpServlet {
  /**
   * Handles GET requests.  Resin will call the doGet method when
   * the browser sends a GET request.
   *
   * @param request the request object contains the data from
   * the browser's request.
   * @param response the response object contains methods to send
   * data back to the browser.
   */
  public void doGet(HttpServletRequest request, HttpServletResponse response)
    throws IOException, ServletException
  {
    // Tells the browser that the data is HTML
    response.setContentType("text/html");
    
    PrintWriter pw = response.getWriter();

    pw.println("<title>URL Request Methods</title>");

    pw.println("<table>");

    // getRequestURL returns the entire URL (minus the query string)
    StringBuffer url = request.getRequestURL();
    
    pw.println("<tr><td>requestURL<td>" + url);

    // The scheme is "http" or "https" for SSL.
    String scheme = request.getScheme();
    
    pw.println("<tr><td>scheme<td>" + scheme);

    // The server name is the name of the virtual host
    String serverName = request.getServerName();
    
    pw.println("<tr><td>serverName<td>" + serverName);

    // Returns the server's TCP port.  Normally 80 or 443 for SSL.
    int serverPort = request.getServerPort();
    
    pw.println("<tr><td>serverPort<td>" + serverPort);

    // Returns the URI, i.e. the entire path after the host
    String uri = request.getRequestURI();
    
    pw.println("<tr><td>requestURI<td>" + uri);

    // Returns the web-app prefix of the URL.
    // In the tutorial, it's /tutorial
    // Servlets belong to web-apps either configured
    // explicitly in the resin.conf or implicitly in
    // the webapps directory.
    String contextPath = request.getContextPath();
    
    pw.println("<tr><td>contextPath<td>" + contextPath);

    // Returns the URL part corresponding to the servlet.
    // For this example, it's /servlet/example.servlet.basic.URLServlet
    String servletPath = request.getServletPath();
    
    pw.println("<tr><td>servletPath<td>" + servletPath);

    // Returns anything after the servletPath in the URL.
    // Servlets will often use the pathInfo as keys into a database
    String pathInfo = request.getPathInfo();
    
    pw.println("<tr><td>pathInfo<td>" + pathInfo);

    // Returns the URL's query string.  Servlets usually use
    // <code>getParameter</code> instead of parsing the query string.
    String queryString = request.getQueryString();
    
    pw.println("<tr><td>queryString<td>" + queryString);
    
    pw.println("</table>");
  }
}

