/*
 * Copyright (c) 1998-2001 Caucho Technology -- all rights reserved
 *
 * Caucho Technology permits redistribution, modification and use
 * of this file in source and binary form ("the Software") under the
 * Caucho Developer Source License ("the License").  The following
 * conditions must be met:
 *
 * 1. Each copy or derived work of the Software must preserve the copyright
 *    notice and this notice unmodified.
 *
 * 2. Redistributions of the Software in source or binary form must include 
 *    an unmodified copy of the License, normally in a plain ASCII text
 *
 * 3. The names "Resin" or "Caucho" are trademarks of Caucho Technology and
 *    may not be used to endorse products derived from this software.
 *    "Resin" or "Caucho" may not appear in the names of products derived
 *    from this software.
 *
 * This Software is provided "AS IS," without a warranty of any kind. 
 * ALL EXPRESS OR IMPLIED REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 *
 * CAUCHO TECHNOLOGY AND ITS LICENSORS SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE OR ANY THIRD PARTY AS A RESULT OF USING OR
 * DISTRIBUTING SOFTWARE. IN NO EVENT WILL CAUCHO OR ITS LICENSORS BE LIABLE
 * FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL,
 * CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND
 * REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
 * INABILITY TO USE SOFTWARE, EVEN IF HE HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGES.      
 */

package example.servlet.basic;

import java.io.*;
import javax.servlet.*;
import javax.servlet.http.*;

/**
 * Shows how the servlet-mapping in the resin.conf and web.xml works.
 *
 * <p>The following is a sample configuration for the examples.  It uses
 * Resin's short syntax which is more scannable and maintainable.
 *
 * <pre>
 * &lt;web-app>
 *
 * &lt;servlet servlet-name='map-a'
 *          servlet-class='example.servlet.basic.MappingServlet'>
 *   &lt;init-param init='value-a'/>
 * &lt;/servlet>
 *
 * &lt;servlet servlet-name='map-b'
 *          servlet-class='example.servlet.basic.MappingServlet'>
 *   &lt;init-param init='value-b'/>
 * &lt;/servlet>
 *
 * &lt;servlet servlet-name='map-c'
 *          servlet-class='example.servlet.basic.MappingServlet'>
 *   &lt;init-param init='value-c'/>
 * &lt;/servlet>
 *
 * &lt;!-- an exact match. -->
 * &lt;servlet-mapping url-pattern='/exact'
 *                  servlet-name='map-a'/>
 *
 * &lt;!-- a prefix match. -->
 * &lt;servlet-mapping url-pattern='/prefix/*'
 *                  servlet-name='map-b'/>
 *
 * &lt;!-- an extension match. -->
 * &lt;servlet-mapping url-pattern='*.map'
 *                  servlet-name='map-c'/>
 *
 * &lt;!-- using the invoker (a prefix match). -->
 * &lt;servlet-mapping url-pattern='/servlet/*'
 *                  servlet-name='invoker'/>
 *
 * &lt;/web-app>
 * </pre>
 *
 * <p>The specific servlet instance depends on the URL.  In the above example,
 * there are three URLs that will use the InitServlet, but all three will
 * use different servlet instances.
 *
 * <table border=0>
 * <tr><td>URL<td>Servlet Path<td>Path Info<td>Servlet
 * <tr><td>/exact<td>/exact<td>null<td>servlet-a
 * <tr><td>/exact/foo<td>n/a<td>n/a<td>404 Not Found
 * <tr><td>/prefix<td>/prefix<td>null<td>servlet-b
 * <tr><td>/prefix/foo<td>/prefix<td>/foo<td>servlet-b
 * <tr><td>/test.map<td>/test.map<td>null<td>servlet-c
 * <tr><td>/exact/test.map<td>/exact/test.map<td>null<td>servlet-c
 * <tr><td>/prefix/test.map<td>/prefix<td>/test.map<td>servlet-b
 * </table>
 *
 * <p>The example includes a id variable so you can see how the servlet
 * instances are reused.  Since Resin only creates a single servlet instance,
 * servlet programmers must be aware of threading issues.
 */
public class MappingServlet extends HttpServlet {
  // A static counting variable so we can number the instances
  private static int count;

  // An integer identifying the instance
  private int id;
  
  // The servlet configuration variable
  private String init;
  
  /**
   * The <code>init()</code> method is called when the servlet is
   * initialized.  For most servlets, it will be called on the first
   * request to the servlet.  For load-on-startup servlets, it will be
   * called when the web-app initializes.
   *
   * <p>Servlets normally use init() to cache initial lookups.  A typical
   * example is caching a lookup of a database DataSource or an EJB home.
   */
  public void init()
    throws ServletException
  {
    // Assign the servlet's id
    id = count++;

    // Gets a servlet initialization parameter
    init = getInitParameter("init");
  }

  /**
   * Handles GET requests.  Resin will call the doGet method when
   * the browser sends a GET request.
   *
   * @param request the request object contains the data from
   * the browser's request.
   * @param response the response object contains methods to send
   * data back to the browser.
   */
  public void doGet(HttpServletRequest request, HttpServletResponse response)
    throws IOException, ServletException
  {
    // Tells the browser that the data is HTML
    response.setContentType("text/html");
    
    PrintWriter pw = response.getWriter();

    pw.println("<title>Mapping Parameters</title>");

    pw.println("<table>");
    
    // The servlet identifier.  We'll print this to show which servlet
    // instances exist.
    pw.println("<tr><td>id<td>" + id);

    // Returns the servlet's init parameter
    pw.println("<tr><td>init<td>" + init);
    
    // Returns the servlet path
    pw.println("<tr><td>servlet-path<td>" + request.getServletPath());
    
    // Returns the path info
    pw.println("<tr><td>path-info<td>" + request.getPathInfo());
    
    pw.println("</table>");
  }
}

